/*
   pi_stress - Priority Inheritance stress test
  
   Copyright (C) 2006 Clark Williams <williams@redhat.com>
  
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301
   USA */

/* This program stress tests pthreads priority inheritance mutexes
`  
   The logic is built upon the state machine that performs the "classic_pi"
   deadlock scenario. 

   That's the theory, anyway... 
   
   CW - 2006  */

#include <stdio.h>
#include <errno.h>
#include <stdlib.h>
#include <stdarg.h>
#include <pthread.h>
#include <sched.h>
#include <unistd.h>
#include <string.h>
#include <signal.h>
#include <getopt.h>
#include <sys/types.h>
#include <sys/wait.h>

// version
const char *version = "pi_stress v0.9 (" __DATE__ ")";

// conversions
#define USEC_PER_SEC 	1000000
#define NSEC_PER_SEC 	1000000000
#define USEC_TO_NSEC(u) ((u) * 1000)
#define USEC_TO_SEC(u) 	((u) / USEC_PER_SEC)
#define NSEC_TO_USEC(n) ((n) / 1000)
#define SEC_TO_NSEC(s) 	((s) * NSEC_PER_SEC)
#define SEC_TO_USEC(s) 	((s) * USEC_PER_SEC)

/* test timeout */
#define TIMEOUT 2

/* determine if the C library supports Priority Inheritance mutexes */
#if defined(_POSIX_THREAD_PRIO_INHERIT) && _POSIX_THREAD_PRIO_INHERIT != -1
#define HAVE_PI_MUTEX 1
#else
#define HAVE_PI_MUTEX 0
#endif

int use_pi_mutex = HAVE_PI_MUTEX;

#define SUCCESS 0
#define FAILURE 1

// the number of groups to create
int ngroups = 10;

// the number of times a group causes a priority inversion situation 
// default to infinite
int inversions = -1;

// turn on lots of prints
int verbose = 0;

int debugging = 0;

// prompt to start test
int prompt = 0;

// report interval
unsigned long report_interval = (unsigned long) SEC_TO_USEC(0.75);

// global that indicates we should shut down
int shutdown = 0;

// command line options
struct option options [] = {
	{ "verbose", no_argument, NULL, 'v' },
	{ "quiet", no_argument, NULL, 'q' },
	{ "no-pi", no_argument, NULL, 'n'}, 
	{ "groups", required_argument, NULL, 'g'},
	{ "inversions" , required_argument, NULL, 'i'},
	{ "report", required_argument, NULL, 'r'},
	{ "fifo", no_argument, NULL, 'f'},
	{ "rr", no_argument, NULL, 'r'},
	{ "prompt", no_argument, NULL, 'p'},
	{ "debug", no_argument, NULL, 'd'},
	{ "version", no_argument, NULL, 'V'},
	{ "help", no_argument, NULL, 'h'},
	{ NULL, 0, NULL, 0},
};

/* define priorities for the threads */
#define MAIN_PRIO(x) (x)
#define RPT_PRIO(x)  (x - 1)
#define HIGH_PRIO(x) (x - 2)
#define MED_PRIO(x)  (x - 3)
#define LOW_PRIO(x)  (x - 4)

#define NUM_TEST_THREADS 3
#define NUM_ADMIN_THREADS 2

#define TIMER_SIGNAL	(SIGRTMIN+1)

pthread_barrier_t all_threads_ready;

cpu_set_t test_cpu_mask, admin_cpu_mask;

int policy = SCHED_FIFO;

struct group_parameters {

	// group id (index)
	int id;

	// threads in the group
	pthread_t low_tid;
	pthread_t med_tid;
	pthread_t high_tid;

	// number of machine iterations to perform
	int inversions;

	// group mutex
	pthread_mutex_t mutex;

        // state barriers
	pthread_barrier_t start_barrier;
	pthread_barrier_t locked_barrier;
	pthread_barrier_t elevate_barrier;
	pthread_barrier_t finish_barrier;

	// state variables
	volatile int high_has_run;
	volatile int low_unlocked;
	volatile int watchdog;

	// total number of inversions performed
	unsigned long total;
} *groups;

/* forward prototypes */
void *low_priority(void *arg);
void *med_priority(void *arg);
void *high_priority(void *arg);
void *reporter(void *arg);
void *watchdog(void *arg);
int setup_thread_attr(pthread_attr_t *attr, int prio, cpu_set_t *mask, int schedpolicy);
int set_cpu_affinity(cpu_set_t *test_mask, cpu_set_t *admin_mask);
void error(char *, ...);
void info(char *, ...);
void debug(char *, ...);

void
usage(void)
{
	printf("usage: pi_stress <options>\n");
	printf("    options:\n");
	printf("\t--quiet\t\t- no output\n");
	printf("\t--verbose\t- lots of output\n");
	printf("\t--no-pi\t\t- don't use PI mutex\n");
	printf("\t--groups=<n>\t- set the number of inversion groups\n");
	printf("\t--inversions=<n>\t- number of inversions per group\n");
	printf("\t--report=<path>\t- output to file\n");
	printf("\t--fifo\t\t- use SCHED_FIFO for test threads\n");
	printf("\t--rr\t\t- use SCHED_RR for test threads\n");
	printf("\t--prompt\t- prompt before starting the test\n");
	printf("\t--debug\t\t- turn on debug prints\n");
	printf("\t--version\t- print version number on output\n");
	printf("\t--help\t\t- print this message\n");
}

// block all signals (called from main)
int
block_signals(void)
{
	int status;
	sigset_t sigset;

	// mask off all signals
	status = sigfillset(&sigset);
	if (status) {
		error("setting up full signal set %s\n", strerror(status));
		return FAILURE;
	}
	status = sigdelset(&sigset, SIGQUIT);
	if (status) {
		error("removing SIGQUIT from signal set: %s\n", strerror(status));
		return FAILURE;
	}
	status = pthread_sigmask(SIG_BLOCK, &sigset, NULL);
	if (status) {
		error("setting signal mask: %s\n", strerror(status));
		return FAILURE;
	}
	return SUCCESS;
}


// clean up on test finish or SIGINT (or an error)
void
cleanup(void)
{
	int i;
	int status;
	struct group_parameters *g;

	// tell anyone that's looking
	shutdown = 1;

	// just whack all the test threads
	for (i = 0; i < ngroups; i++) {
		g = &groups[i];
		if (g->low_tid && (status = pthread_kill(g->low_tid, SIGQUIT)))
			error("sending SIGQUIT to groups[%d].low_tid: %s\n", 
			      i, strerror(status));
		if (g->med_tid && (status = pthread_kill(g->med_tid, SIGQUIT)))
			error("sending SIGQUIT to groups[%d].med_tid: %s\n", 
			      i, strerror(status));
		if (g->high_tid && (status = pthread_kill(g->high_tid, SIGQUIT)))
			error("sending SIGQUIT to groups[%d].high_tid: %s\n",
			      i, strerror(status));
	}
}

// set up a test group
int
initialize_group(struct group_parameters *group)
{
	int status;
	pthread_mutexattr_t mutex_attr;

	group->inversions = inversions;

	// setup default attributes for the group mutex
	// (make it a PI mutex)
	status = pthread_mutexattr_init(&mutex_attr);
	if (status) {
		error("initializing mutex attribute: %s\n", strerror(status));
		return FAILURE;
	}

	if (use_pi_mutex) {
		/* set priority inheritance attribute for mutex */
		status = pthread_mutexattr_setprotocol(&mutex_attr, 
						       PTHREAD_PRIO_INHERIT);
		if (status) {
			error("setting mutex attribute policy: %s\n", strerror(status));
			return FAILURE;
		}
	}

	// initialize the group mutex
	status = pthread_mutex_init(&group->mutex, &mutex_attr);
	if (status) {
		error("initializing mutex: %s\n", strerror(status));
		return FAILURE;
	}

	// initialize the group barriers
	status = pthread_barrier_init(&group->start_barrier, NULL, NUM_TEST_THREADS);
	if (status) {
		error("failed to initialize start_barrier\n");
		return FAILURE;
	}
	status = pthread_barrier_init(&group->locked_barrier, NULL, 2);
	if (status) {
		error("failed to intialize locked_barrier: %s\n", strerror(status));
		return FAILURE;
	}
	status = pthread_barrier_init(&group->elevate_barrier, NULL, 2);
	if (status) {
		error("failed to initialize elevate_barrier: %s\n", strerror(status));
		return FAILURE;
	}
	status = pthread_barrier_init(&group->finish_barrier, NULL, NUM_TEST_THREADS);
	if (status) {
		error("failed to initialize finish_barrier: %s\n", strerror(status));
		return FAILURE;
	}
	return SUCCESS;
}	
// setup and create a groups threads
int
create_group(struct group_parameters *group, cpu_set_t *mask)
{
	int status;
	int prio_max = sched_get_priority_max(policy);
	pthread_attr_t thread_attr;

	// initialize group structure
	status = initialize_group(group);
	if (status) {
		error("initializing group %d\n", group->id);
		return FAILURE;
	}

	/* start the low priority thread */
	debug("creating low priority thread\n");
	if (setup_thread_attr(&thread_attr, LOW_PRIO(prio_max), mask, policy))
		return FAILURE;
	status = pthread_create(&group->low_tid, 
				&thread_attr, 
				low_priority, 
				group);
	if (status != 0) {
		error("creating low_priority thread: %s\n", strerror(status));
		return FAILURE;
	}

	/* create the medium priority thread */
	debug("creating medium priority thread\n");
	if (setup_thread_attr(&thread_attr, MED_PRIO(prio_max), mask, policy))
		return FAILURE;
	status = pthread_create(&group->med_tid,
				&thread_attr, 
				med_priority, 
				group);
	if (status != 0) {
		error("creating med_priority thread: %s\n", strerror(status));
		return FAILURE;
	}

	/* create the high priority thread */
	debug("creating high priority thread\n");
	if (setup_thread_attr(&thread_attr, HIGH_PRIO(prio_max), mask, policy))
		return FAILURE;
	status = pthread_create(&group->high_tid,
				&thread_attr, 
				high_priority, 
				group);
	if (status != 0) {
		error("creating high_priority thread: %s\n", strerror(status));
		cleanup();
		return FAILURE;
	}
	return SUCCESS;
}

void
process_command_line(int argc, char **argv)
{
	int opt;
	while ((opt = getopt_long(argc, argv, "+", options, NULL)) != -1) {
		switch (opt) {
		case '?':
		case 'h':
			usage();
			exit(0);
		case 'v':
			verbose = 1;
			break;
		case 'q':
			verbose = 0;
			break;
		case 'n':
			use_pi_mutex = 0;
			break;
		case 'i':
			inversions = strtol(optarg, NULL, 10);
			info("doing %d inversion per group\n", inversions);
			break;
		case 'g':
			ngroups = strtol(optarg, NULL, 10);
			info("number of groups set to %d\n", ngroups);
			break;
		case 'f':
			policy = SCHED_FIFO;
			break;
		case 'r':
			policy = SCHED_RR;
			break;
		case 'p':
			prompt = 1;
			break;
		case 'd':
			debugging = 1;
			break;
		case 'V':
			puts(version);
			exit(0);
		}
	}
}

// total the number of inversions that have been performed
unsigned long
total_inversions()
{
	int i;
	unsigned long total = 0;

	for (i = 0; i < ngroups; i++)
		total += groups[i].total;
	return total;
}

int
main(int argc, char **argv)
{
	int status;
	int prio_max;
	struct sched_param thread_param;
	int i;
	int signo;
	sigset_t sigset;
	struct sigaction sa;
	pthread_t report_thread;
	pthread_attr_t thread_attr;

	/* Make sure we see all message, even those on stdout.  */
	setvbuf (stdout, NULL, _IONBF, 0);

	/* process command line arguments */
	process_command_line(argc, argv);
	info("Number of groups: %d\n", ngroups);
	if (inversions < 0)
		info("Number of inversions per group: infinite\n");
	else
		info("Number of inversions per group: %d\n", inversions);
	info("Priority Inheritance turned %s\n", use_pi_mutex ? "on" : "off");

	prio_max = sched_get_priority_max(policy);

	// boost main to max priority (so we keep running) :)
	thread_param.sched_priority = MAIN_PRIO(sched_get_priority_max(SCHED_FIFO));
	status = pthread_setschedparam(pthread_self(), SCHED_FIFO, &thread_param);
	if (status) {
		error("main: boosting to max priority: 0x%x\n", status);
		return FAILURE;
	}
	info("Main thread priority: %d\n", MAIN_PRIO(prio_max));
	info("High priority: %d\n", HIGH_PRIO(prio_max));
	info("Med priority:  %d\n", MED_PRIO(prio_max));
	info("Low priority:  %d\n", LOW_PRIO(prio_max));

	// block unwanted signals
	block_signals();

	// allocate our groups array
	groups = calloc(ngroups, sizeof(struct group_parameters));
	if (groups == NULL) {
		error("main: failed to allocate %d groups\n", ngroups);
		return FAILURE;
	}

	// set up CPU affinity masks
	if (set_cpu_affinity(&test_cpu_mask, &admin_cpu_mask))
		return FAILURE;
		
	// set up our ready barrier
	status = pthread_barrier_init(&all_threads_ready, NULL, 
				      (ngroups * NUM_TEST_THREADS) + NUM_ADMIN_THREADS);
	if (status) {
		error("initialize_barriers: failed to initialize all_threads_ready\n");
		return FAILURE;
	}

	// create the groups
	info("Creating %d test groups\n", ngroups);
	for (i = 0; i < ngroups; i++) {
		groups[i].id = i;
		if (create_group(&groups[i], &test_cpu_mask))
			return FAILURE;
	}

	// start our reporter thread
	status = setup_thread_attr(&thread_attr, RPT_PRIO(prio_max), &admin_cpu_mask, SCHED_FIFO);
	if (status) {
		error("setting report thread attributes: %s\n", strerror(status));
		cleanup();
		return FAILURE;
	}
	status = pthread_create(&report_thread, &thread_attr, reporter, NULL);
	if (status) {
		error("creating reporter thread: %s\n", strerror(status));
		cleanup();
		return FAILURE;
	}

	// prompt if requested
	if (prompt) {
		printf("Press return to start test: ");
		getchar();
	}

	// set the SIGCLD action to SIG_DFL (rather than SIG_IGN)
	status = sigemptyset(&sigset);
	if (status) {
		error("failed to create empty signal set: %s\n", strerror(errno));
		cleanup();
		return FAILURE;
	}
	status = sigaddset(&sigset, SIGCLD);
	if (status) {
		error("failed to add SIGCLD to signal set: %s\n", strerror(errno));
		cleanup();
		return FAILURE;
	}
	sa.sa_handler = SIG_DFL;
	sa.sa_mask = sigset;
	sa.sa_flags = 0;
	sa.sa_sigaction = NULL;
	status = sigaction(SIGCLD, &sa, NULL);
	if (status) {
		error("setting signal action for SIGCLD: %s\n", strerror(errno));
		cleanup();
		return FAILURE;
	}

	// add in SIGINT so we can wait for either SIGCLD or SIGINT
	status = sigaddset(&sigset, SIGINT);
	if (status) {
		error("failed to add SIGINT to signal set: %s\n", strerror(errno));
		cleanup();
		return FAILURE;
	}
	
	// turn loose the threads
	info("Releasing all threads\n");
	status = pthread_barrier_wait(&all_threads_ready);
	if (status && status != PTHREAD_BARRIER_SERIAL_THREAD) {
		error("main: pthread_barrier_wait(all_threads_ready): 0x%x\n", status);
		cleanup();
		return FAILURE;
	}

	// wait for either an interrupt or the reporter to die
	printf("Press Ctrl-C to stop test\n");
	status = sigwait(&sigset, &signo);
	if (status)
		error("during sigwait: %s\n", strerror(status));
	cleanup();
	info("All threads terminated!\n");
	return SUCCESS;
}


int setup_thread_attr(pthread_attr_t *attr, int prio, cpu_set_t *mask, int schedpolicy)
{
	int status;
	struct sched_param thread_param;

	status = pthread_attr_init(attr);
	if (status) {
		error("setup_thread_attr: initializing thread attribute: 0x%x\n", status);
		return FAILURE;
	}
	status = pthread_attr_setschedpolicy(attr, schedpolicy);
	if (status) {
		error("setup_thread_attr: setting attribute policy to %s: 0x%x\n", 
		      schedpolicy == SCHED_FIFO ? "SCHED_FIFO" : "SCHED_RR", status);
		return FAILURE;
	}
	status = pthread_attr_setinheritsched(attr, PTHREAD_EXPLICIT_SCHED);
	if (status) {
		error("setup_thread_attr: setting explicit scheduling inheritance: 0x%x\n", status);
		return FAILURE;
	}
	thread_param.sched_priority = prio;
	status = pthread_attr_setschedparam(attr, &thread_param);
	if (status) {
		error("setup_thread_attr: setting scheduler param: 0x%x\n", status);
		return FAILURE;
	}
	status = pthread_attr_setaffinity_np(attr, sizeof(cpu_set_t), mask);
	if (status) {
		error("setup_thread_attr: setting affinity attribute: 0x%x\n", status);
		return FAILURE;
	}
	return SUCCESS;
}

int set_cpu_affinity(cpu_set_t *test_mask, cpu_set_t *admin_mask)
{
	int status, i;
	cpu_set_t current_mask;
	int max_processors = sizeof(cpu_set_t) * 8;

	// first set our main thread to run on cpu zero
	status = sched_getaffinity(0, sizeof(cpu_set_t), &current_mask);
	if (status) {
		error("failed getting CPU affinity mask: 0x%x\n", status);
		return FAILURE;
	}
	for (i = 0; i < max_processors; i++) {
		if (CPU_ISSET(i, &current_mask))
		break;
	}
	if (i >= max_processors) {
		error("No schedulable CPU found for main!\n");
		return FAILURE;
	}
	CPU_ZERO(admin_mask);
	CPU_SET(i, admin_mask);
	status = sched_setaffinity(0, sizeof(cpu_set_t), admin_mask);
	if (status) {
		error("set_cpu_affinity: setting CPU affinity mask: 0x%x\n", status);
		return FAILURE;
	}
	info("Main thread will run on processor %d\n", i);

	/* Now set up a CPU affinity mask so that tests only run on one processor */
	for (i = max_processors - 1; i >= 0; i--) {
		if (CPU_ISSET(i, &current_mask))
			break;
	}
	if (i < 0) {
		error("No schedulable CPU found for tests!\n");
		return FAILURE;
	}
	CPU_ZERO(test_mask);
	CPU_SET(i, test_mask);
	info("Tests will run on processor %d\n", i);
	return SUCCESS;
}

int
report_threadinfo(char *name)
{
	int status;
	struct sched_param thread_param;
	int thread_policy;

	status = pthread_getschedparam(pthread_self(), &thread_policy, &thread_param);
	if (status) {
		error("report_threadinfo: failed to get scheduler param: 0x%x\n", status);
		return FAILURE;
	}
	debug("%s: running as %s thread at priority %d\n",
	     name, thread_policy == SCHED_FIFO ? "FIFO" :
	     thread_policy == SCHED_RR ? "RR" : "OTHER",
	     thread_param.sched_priority);
	return SUCCESS;
}

void watchdog_clear(void)
{
	int i;
	for (i = 0; i < ngroups; i++)
		groups[i].watchdog = 0;
}

int watchdog_check(void)
{
	int i;
	for (i = 0; i < ngroups; i++)
		if (groups[i].watchdog == 0) {
			error("WATCHDOG triggered: group %d is deadlocked!\n", i);
			return i;
		}
	return SUCCESS;
}	
	
void *reporter(void *arg)
{
	int group;
	struct timespec ts;
	int status;

	ts.tv_sec = 0;
	ts.tv_nsec = USEC_TO_NSEC(report_interval);

	// wait for all threads to be ready
	status = pthread_barrier_wait(&all_threads_ready);
	if (status && status != PTHREAD_BARRIER_SERIAL_THREAD) {
		error("reporter: pthread_barrier_wait(all_threads_ready): %x", status);
		return NULL;
	}

	while (shutdown == 0) {
		// clear watchdog counters
		watchdog_clear();

		// wait for our reporting interval
		status = clock_nanosleep(CLOCK_MONOTONIC, 0, &ts, NULL);
		if (status) {
			error("from clock_nanosleep: %s\n", strerror(status));
			return NULL;
		}
		printf("Current Inversions: %lu\n", total_inversions());
		fputs("\033[1A", stdout);

		// check watchdog stuff
		if ((group = watchdog_check())) {
			error("reporter stopping due to watchdog event on group %d\n", group);
			cleanup();
		}
			
	}
	fputs("\n\n\033[1B", stdout);
	printf("Total Inversions: %lu\n", total_inversions());
	return NULL;
}


void *low_priority(void *arg)
{
	int status;
	int unbounded;
	unsigned long count = 0;
	struct group_parameters *p = (struct group_parameters *)arg;

	if (report_threadinfo("low_priority"))
		return NULL;

	debug("low_priority: entering ready state\n");

	/* wait for all threads to be ready */
	status = pthread_barrier_wait(&all_threads_ready);
	if (status && status != PTHREAD_BARRIER_SERIAL_THREAD) {
		error("low_priority: pthread_barrier_wait(all_threads_ready): %x", status);
		return NULL;
	}

	unbounded = (p->inversions < 0);

	debug("low_priority: starting inversion loop\n");
	while (unbounded || p->inversions-- > 0) {
		/* initial state */
		debug("low_priority: entering start wait (%d)\n", count++);
		status = pthread_barrier_wait(&p->start_barrier);
		if (status && status != PTHREAD_BARRIER_SERIAL_THREAD) {
			error("low_priority: pthread_barrier_wait(start): %x\n", status);
			return NULL;
		}
		debug("low_priority: claiming mutex\n");
		pthread_mutex_lock(&p->mutex);
		debug("low_priority: mutex locked\n");

		debug("low_priority: entering locked wait\n");
		status = pthread_barrier_wait(&p->locked_barrier);
		if (status && status != PTHREAD_BARRIER_SERIAL_THREAD) {
			error("low_priority: pthread_barrier_wait(locked): %x\n", status);
			return NULL;
		}

		/* wait for priority boost */
		debug("low_priority: entering elevated wait\n");
		p->low_unlocked = 0; /* prevent race with med_priority */
		status = pthread_barrier_wait(&p->elevate_barrier);
		if (status && status != PTHREAD_BARRIER_SERIAL_THREAD) {
			error("low_priority: pthread_barrier_wait(elevate): %x\n", status);
			return NULL;
		}
		p->low_unlocked = 1;

		/* release the mutex */
		debug("low_priority: unlocking mutex\n");
		pthread_mutex_unlock(&p->mutex);

		/* finish state */
		debug("low_priority: entering finish wait\n");
		status = pthread_barrier_wait(&p->finish_barrier);
		if (status && status != PTHREAD_BARRIER_SERIAL_THREAD) {
			error("low_priority: pthread_barrier_wait(elevate): %x\n", status);
			return NULL;
		}
		
	}
	// exit
	debug("low_priority: exiting\n");
	return NULL;
}

void *med_priority(void *arg)
{
	int status;
	int unbounded;
	unsigned long count = 0;
	struct group_parameters *p = (struct group_parameters *)arg;

	if (report_threadinfo("med_priority"))
		return NULL;

	debug("med_priority: entering ready state\n");
	/* wait for all threads to be ready */
	status = pthread_barrier_wait(&all_threads_ready);
	if (status && status != PTHREAD_BARRIER_SERIAL_THREAD) {
		error("med_priority: pthread_barrier_wait(all_threads_ready): %x", status);
		return NULL;
	}

	unbounded = (p->inversions < 0);

	debug("med_priority: starting inversion loop\n");
	while (unbounded || p->inversions-- > 0) {
		/* start state */
		debug("med_priority: entering start state (%d)\n", count++);
		status = pthread_barrier_wait(&p->start_barrier);
		if (status && status != PTHREAD_BARRIER_SERIAL_THREAD) {
			error("med_priority: pthread_barrier_wait(start): %x", status);
			return NULL;
		}
		debug("med_priority: entering elevate state\n");
		do {
			status = pthread_barrier_wait(&p->elevate_barrier);
			if (status && status != PTHREAD_BARRIER_SERIAL_THREAD) {
				error("med_priority: pthread_barrier_wait(elevate): %x", status);
				return NULL;
			}
		} while (!p->high_has_run && !p->low_unlocked);
		debug("med_priority: entering finish state\n");
		status = pthread_barrier_wait(&p->finish_barrier);
		if (status && status != PTHREAD_BARRIER_SERIAL_THREAD) {
			error("med_priority: pthread_barrier_wait(finished): %x", status);
			return NULL;
		}
	}

	// exit
	debug("med_priority: exiting\n");
	return NULL;
}

void *high_priority(void *arg)
{
	int status;
	int unbounded;
	unsigned long count = 0;
	struct group_parameters *p = (struct group_parameters *)arg;

	if (report_threadinfo("high_priority"))
		return NULL;

	debug("high_priority: entering ready state\n");

	/* wait for all threads to be ready */
	status = pthread_barrier_wait(&all_threads_ready);
	if (status && status != PTHREAD_BARRIER_SERIAL_THREAD) {
		error("high_priority: pthread_barrier_wait(all_threads_ready): %x", status);
		return NULL;
	}
	unbounded = (p->inversions < 0);
	debug("high_priority: starting inversion loop\n");
	while (unbounded || p->inversions-- > 0) {
		p->high_has_run = 0;
		debug("high_priority: entering start state (%d)\n", count++);
		status = pthread_barrier_wait(&p->start_barrier);
		if (status && status != PTHREAD_BARRIER_SERIAL_THREAD) {
			error("high_priority: pthread_barrier_wait(start): %x", status);
			return NULL;
		}
		debug("high_priority: entering running state\n");
		status = pthread_barrier_wait(&p->locked_barrier);
		if (status && status != PTHREAD_BARRIER_SERIAL_THREAD) {
			error("high_priority: pthread_barrier_wait(running): %x", status);
			return NULL;
		}
		debug("high_priority: locking mutex\n");
		pthread_mutex_lock(&p->mutex);
		debug("high_priority: got mutex\n");
		p->high_has_run = 1;
		debug("high_priority: unlocking mutex\n");
		pthread_mutex_unlock(&p->mutex);
		debug("high_priority: entering finish state\n");
		status = pthread_barrier_wait(&p->finish_barrier);
		if (status && status != PTHREAD_BARRIER_SERIAL_THREAD) {
			error("high_priority: pthread_barrier_wait(finish): %x", status);
			return NULL;
		}

		// update the group stats
		p->total++;

		// update the watchdog counter
		p->watchdog++;
	}
       
	// exit
	debug("high_priority: exiting\n");	
	return NULL;
}

void error(char *fmt, ...)
{
	va_list ap;
	fputs("ERROR: ", stderr);
	va_start(ap, fmt);
	vfprintf(stderr, fmt, ap);
	va_end(ap);
}

void info(char *fmt, ...)
{
	if (verbose) {
		va_list ap;
		va_start(ap, fmt);
		vprintf(fmt, ap);
		va_end(ap);
	}
}

void debug(char *fmt, ...)
{
	if (debugging) {
		va_list ap;
		fputs("DEBUG: ", stderr);
		va_start(ap, fmt);
		vfprintf(stderr, fmt, ap);
		va_end(ap);
	}
}
	
