/*
 * Copyright 2020 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */


package android.app.appsearch;

import java.util.Map;
import android.app.appsearch.exceptions.AppSearchException;
import java.util.List;

/**
 * This class represents the specification logic for AppSearch. It can be used to set the type of
 * search, like prefix or exact only or apply filters to search for a specific schema type only etc.
 */

@SuppressWarnings({"unchecked", "deprecation", "all"})
public final class SearchSpec {

SearchSpec() { throw new RuntimeException("Stub!"); }

/**
 * Returns how the query terms should match terms in the index.
 * @return Value is {@link android.app.appsearch.SearchSpec#TERM_MATCH_EXACT_ONLY}, or {@link android.app.appsearch.SearchSpec#TERM_MATCH_PREFIX}
 */

public int getTermMatch() { throw new RuntimeException("Stub!"); }

/**
 * Returns the list of schema types to search for.
 *
 * <p>If empty, the query will search over all schema types.

 * @return This value cannot be {@code null}.
 */

@android.annotation.NonNull
public java.util.List<java.lang.String> getFilterSchemas() { throw new RuntimeException("Stub!"); }

/**
 * Returns the list of namespaces to search over.
 *
 * <p>If empty, the query will search over all namespaces.

 * @return This value cannot be {@code null}.
 */

@android.annotation.NonNull
public java.util.List<java.lang.String> getFilterNamespaces() { throw new RuntimeException("Stub!"); }

/**
 * Returns the list of package name filters to search over.
 *
 * <p>If empty, the query will search over all packages that the caller has access to. If
 * package names are specified which caller doesn't have access to, then those package names
 * will be ignored.

 * @return This value cannot be {@code null}.
 */

@android.annotation.NonNull
public java.util.List<java.lang.String> getFilterPackageNames() { throw new RuntimeException("Stub!"); }

/** Returns the number of results per page in the result set. */

public int getResultCountPerPage() { throw new RuntimeException("Stub!"); }

/**
 * Returns the ranking strategy.
 * @return Value is {@link android.app.appsearch.SearchSpec#RANKING_STRATEGY_NONE}, {@link android.app.appsearch.SearchSpec#RANKING_STRATEGY_DOCUMENT_SCORE}, {@link android.app.appsearch.SearchSpec#RANKING_STRATEGY_CREATION_TIMESTAMP}, {@link android.app.appsearch.SearchSpec#RANKING_STRATEGY_RELEVANCE_SCORE}, {@link android.app.appsearch.SearchSpec#RANKING_STRATEGY_USAGE_COUNT}, {@link android.app.appsearch.SearchSpec#RANKING_STRATEGY_USAGE_LAST_USED_TIMESTAMP}, {@link android.app.appsearch.SearchSpec#RANKING_STRATEGY_SYSTEM_USAGE_COUNT}, {@link android.app.appsearch.SearchSpec#RANKING_STRATEGY_SYSTEM_USAGE_LAST_USED_TIMESTAMP}, {@link android.app.appsearch.SearchSpec#RANKING_STRATEGY_JOIN_AGGREGATE_SCORE}, or {@link android.app.appsearch.SearchSpec#RANKING_STRATEGY_ADVANCED_RANKING_EXPRESSION}
 */

public int getRankingStrategy() { throw new RuntimeException("Stub!"); }

/**
 * Returns the order of returned search results (descending or ascending).
 * @return Value is {@link android.app.appsearch.SearchSpec#ORDER_DESCENDING}, or {@link android.app.appsearch.SearchSpec#ORDER_ASCENDING}
 */

public int getOrder() { throw new RuntimeException("Stub!"); }

/** Returns how many documents to generate snippets for. */

public int getSnippetCount() { throw new RuntimeException("Stub!"); }

/**
 * Returns how many matches for each property of a matching document to generate snippets for.
 */

public int getSnippetCountPerProperty() { throw new RuntimeException("Stub!"); }

/** Returns the maximum size of a snippet in characters. */

public int getMaxSnippetSize() { throw new RuntimeException("Stub!"); }

/**
 * Returns a map from schema type to property paths to be used for projection.
 *
 * <p>If the map is empty, then all properties will be retrieved for all results.
 *
 * <p>Calling this function repeatedly is inefficient. Prefer to retain the Map returned by this
 * function, rather than calling it multiple times.
 *
 * @return A mapping of schema types to lists of projection strings.

 * This value cannot be {@code null}.
 */

@android.annotation.NonNull
public java.util.Map<java.lang.String,java.util.List<java.lang.String>> getProjections() { throw new RuntimeException("Stub!"); }

/**
 * Returns a map from schema type to property paths to be used for projection.
 *
 * <p>If the map is empty, then all properties will be retrieved for all results.
 *
 * <p>Calling this function repeatedly is inefficient. Prefer to retain the Map returned by this
 * function, rather than calling it multiple times.
 *
 * @return A mapping of schema types to lists of projection {@link android.app.appsearch.PropertyPath PropertyPath} objects.

 * This value cannot be {@code null}.
 */

@android.annotation.NonNull
public java.util.Map<java.lang.String,java.util.List<android.app.appsearch.PropertyPath>> getProjectionPaths() { throw new RuntimeException("Stub!"); }

/**
 * Returns properties weights to be used for scoring.
 *
 * <p>Calling this function repeatedly is inefficient. Prefer to retain the {@link java.util.Map Map} returned
 * by this function, rather than calling it multiple times.
 *
 * @return a {@link java.util.Map Map} of schema type to an inner-map of property paths of the schema type to
 *     the weight to set for that property.

 * This value cannot be {@code null}.
 */

@android.annotation.NonNull
public java.util.Map<java.lang.String,java.util.Map<java.lang.String,java.lang.Double>> getPropertyWeights() { throw new RuntimeException("Stub!"); }

/**
 * Returns properties weights to be used for scoring.
 *
 * <p>Calling this function repeatedly is inefficient. Prefer to retain the {@link java.util.Map Map} returned
 * by this function, rather than calling it multiple times.
 *
 * @return a {@link java.util.Map Map} of schema type to an inner-map of property paths of the schema type to
 *     the weight to set for that property.

 * This value cannot be {@code null}.
 */

@android.annotation.NonNull
public java.util.Map<java.lang.String,java.util.Map<android.app.appsearch.PropertyPath,java.lang.Double>> getPropertyWeightPaths() { throw new RuntimeException("Stub!"); }

/**
 * Get the type of grouping limit to apply, or 0 if {@link android.app.appsearch.SearchSpec.Builder#setResultGrouping Builder#setResultGrouping} was not
 * called.

 * @return Value is either <code>0</code> or a combination of {@link android.app.appsearch.SearchSpec#GROUPING_TYPE_PER_PACKAGE}, {@link android.app.appsearch.SearchSpec#GROUPING_TYPE_PER_NAMESPACE}, and android.app.appsearch.SearchSpec.GROUPING_TYPE_PER_SCHEMA
 */

public int getResultGroupingTypeFlags() { throw new RuntimeException("Stub!"); }

/**
 * Get the maximum number of results to return for each group.
 *
 * @return the maximum number of results to return for each group or Integer.MAX_VALUE if {@link android.app.appsearch.SearchSpec.Builder#setResultGrouping(int,int)      } was not called.
 */

public int getResultGroupingLimit() { throw new RuntimeException("Stub!"); }

/**
 * Returns specification on which documents need to be joined.
 * @return This value may be {@code null}.
 */

@android.annotation.Nullable
public android.app.appsearch.JoinSpec getJoinSpec() { throw new RuntimeException("Stub!"); }

/**
 * Get the advanced ranking expression, or "" if {@link android.app.appsearch.SearchSpec.Builder#setRankingStrategy(java.lang.String) Builder#setRankingStrategy(String)} was
 * not called.

 * @return This value cannot be {@code null}.
 */

@android.annotation.NonNull
public java.lang.String getAdvancedRankingExpression() { throw new RuntimeException("Stub!"); }

/** Returns whether the {@link Features#NUMERIC_SEARCH} feature is enabled. */

public boolean isNumericSearchEnabled() { throw new RuntimeException("Stub!"); }

/** Returns whether the {@link Features#VERBATIM_SEARCH} feature is enabled. */

public boolean isVerbatimSearchEnabled() { throw new RuntimeException("Stub!"); }

/** Returns whether the {@link Features#LIST_FILTER_QUERY_LANGUAGE} feature is enabled. */

public boolean isListFilterQueryLanguageEnabled() { throw new RuntimeException("Stub!"); }

/**
 * Results should be grouped together by namespace for the purpose of enforcing a limit on the
 * number of results returned per namespace.
 */

public static final int GROUPING_TYPE_PER_NAMESPACE = 2; // 0x2

/**
 * Results should be grouped together by package for the purpose of enforcing a limit on the
 * number of results returned per package.
 */

public static final int GROUPING_TYPE_PER_PACKAGE = 1; // 0x1

/** Search results will be returned in an ascending order. */

public static final int ORDER_ASCENDING = 1; // 0x1

/** Search results will be returned in a descending order. */

public static final int ORDER_DESCENDING = 0; // 0x0

/**
 * Schema type to be used in {@link android.app.appsearch.SearchSpec.Builder#addProjection SearchSpec.Builder#addProjection} to apply property paths to
 * all results, excepting any types that have had their own, specific property paths set.
 */

public static final java.lang.String PROJECTION_SCHEMA_TYPE_WILDCARD = "*";

/** Ranked by the advanced ranking expression provided. */

public static final int RANKING_STRATEGY_ADVANCED_RANKING_EXPRESSION = 9; // 0x9

/** Ranked by document creation timestamps. */

public static final int RANKING_STRATEGY_CREATION_TIMESTAMP = 2; // 0x2

/** Ranked by app-provided document scores. */

public static final int RANKING_STRATEGY_DOCUMENT_SCORE = 1; // 0x1

/**
 * Ranked by the aggregated ranking signal of the joined documents.
 *
 * <p>Which aggregation strategy is used to determine a ranking signal is specified in the
 * {@link android.app.appsearch.JoinSpec JoinSpec} set by {@link android.app.appsearch.SearchSpec.Builder#setJoinSpec Builder#setJoinSpec}. This ranking strategy may not be used if
 * no {@link android.app.appsearch.JoinSpec JoinSpec} is provided.
 *
 * @see android.app.appsearch.SearchSpec.Builder#build
 */

public static final int RANKING_STRATEGY_JOIN_AGGREGATE_SCORE = 8; // 0x8

/** No Ranking, results are returned in arbitrary order. */

public static final int RANKING_STRATEGY_NONE = 0; // 0x0

/** Ranked by document relevance score. */

public static final int RANKING_STRATEGY_RELEVANCE_SCORE = 3; // 0x3

/** Ranked by number of usages from a system UI surface. */

public static final int RANKING_STRATEGY_SYSTEM_USAGE_COUNT = 6; // 0x6

/** Ranked by timestamp of last usage from a system UI surface. */

public static final int RANKING_STRATEGY_SYSTEM_USAGE_LAST_USED_TIMESTAMP = 7; // 0x7

/** Ranked by number of usages, as reported by the app. */

public static final int RANKING_STRATEGY_USAGE_COUNT = 4; // 0x4

/** Ranked by timestamp of last usage, as reported by the app. */

public static final int RANKING_STRATEGY_USAGE_LAST_USED_TIMESTAMP = 5; // 0x5

/**
 * Query terms will only match exact tokens in the index.
 *
 * <p>Ex. A query term "foo" will only match indexed token "foo", and not "foot" or "football".
 */

public static final int TERM_MATCH_EXACT_ONLY = 1; // 0x1

/**
 * Query terms will match indexed tokens when the query term is a prefix of the token.
 *
 * <p>Ex. A query term "foo" will match indexed tokens like "foo", "foot", and "football".
 */

public static final int TERM_MATCH_PREFIX = 2; // 0x2
/** Builder for {@link android.app.appsearch.SearchSpec objects}. */

@SuppressWarnings({"unchecked", "deprecation", "all"})
public static final class Builder {

public Builder() { throw new RuntimeException("Stub!"); }

/**
 * Indicates how the query terms should match {@code TermMatchCode} in the index.
 *
 * <p>If this method is not called, the default term match type is {@link android.app.appsearch.SearchSpec#TERM_MATCH_PREFIX  }.
 
 * @param termMatchType Value is {@link android.app.appsearch.SearchSpec#TERM_MATCH_EXACT_ONLY}, or {@link android.app.appsearch.SearchSpec#TERM_MATCH_PREFIX}
 * @return This value cannot be {@code null}.
 */

@android.annotation.NonNull
public android.app.appsearch.SearchSpec.Builder setTermMatch(int termMatchType) { throw new RuntimeException("Stub!"); }

/**
 * Adds a Schema type filter to {@link android.app.appsearch.SearchSpec SearchSpec} Entry. Only search for documents that
 * have the specified schema types.
 *
 * <p>If unset, the query will search over all schema types.
 
 * @param schemas This value cannot be {@code null}.
 * @return This value cannot be {@code null}.
 */

@android.annotation.NonNull
public android.app.appsearch.SearchSpec.Builder addFilterSchemas(@android.annotation.NonNull java.lang.String... schemas) { throw new RuntimeException("Stub!"); }

/**
 * Adds a Schema type filter to {@link android.app.appsearch.SearchSpec SearchSpec} Entry. Only search for documents that
 * have the specified schema types.
 *
 * <p>If unset, the query will search over all schema types.
 
 * @param schemas This value cannot be {@code null}.
 * @return This value cannot be {@code null}.
 */

@android.annotation.NonNull
public android.app.appsearch.SearchSpec.Builder addFilterSchemas(@android.annotation.NonNull java.util.Collection<java.lang.String> schemas) { throw new RuntimeException("Stub!"); }

/**
 * Adds a namespace filter to {@link android.app.appsearch.SearchSpec SearchSpec} Entry. Only search for documents that have
 * the specified namespaces.
 *
 * <p>If unset, the query will search over all namespaces.
 
 * @param namespaces This value cannot be {@code null}.
 * @return This value cannot be {@code null}.
 */

@android.annotation.NonNull
public android.app.appsearch.SearchSpec.Builder addFilterNamespaces(@android.annotation.NonNull java.lang.String... namespaces) { throw new RuntimeException("Stub!"); }

/**
 * Adds a namespace filter to {@link android.app.appsearch.SearchSpec SearchSpec} Entry. Only search for documents that have
 * the specified namespaces.
 *
 * <p>If unset, the query will search over all namespaces.
 
 * @param namespaces This value cannot be {@code null}.
 * @return This value cannot be {@code null}.
 */

@android.annotation.NonNull
public android.app.appsearch.SearchSpec.Builder addFilterNamespaces(@android.annotation.NonNull java.util.Collection<java.lang.String> namespaces) { throw new RuntimeException("Stub!"); }

/**
 * Adds a package name filter to {@link android.app.appsearch.SearchSpec SearchSpec} Entry. Only search for documents that
 * were indexed from the specified packages.
 *
 * <p>If unset, the query will search over all packages that the caller has access to. If
 * package names are specified which caller doesn't have access to, then those package names
 * will be ignored.
 
 * @param packageNames This value cannot be {@code null}.
 * @return This value cannot be {@code null}.
 */

@android.annotation.NonNull
public android.app.appsearch.SearchSpec.Builder addFilterPackageNames(@android.annotation.NonNull java.lang.String... packageNames) { throw new RuntimeException("Stub!"); }

/**
 * Adds a package name filter to {@link android.app.appsearch.SearchSpec SearchSpec} Entry. Only search for documents that
 * were indexed from the specified packages.
 *
 * <p>If unset, the query will search over all packages that the caller has access to. If
 * package names are specified which caller doesn't have access to, then those package names
 * will be ignored.
 
 * @param packageNames This value cannot be {@code null}.
 * @return This value cannot be {@code null}.
 */

@android.annotation.NonNull
public android.app.appsearch.SearchSpec.Builder addFilterPackageNames(@android.annotation.NonNull java.util.Collection<java.lang.String> packageNames) { throw new RuntimeException("Stub!"); }

/**
 * Sets the number of results per page in the returned object.
 *
 * <p>The default number of results per page is 10.
 
 * @param resultCountPerPage Value is between 0 and MAX_NUM_PER_PAGE inclusive
 * @return This value cannot be {@code null}.
 */

@android.annotation.NonNull
public android.app.appsearch.SearchSpec.Builder setResultCountPerPage(int resultCountPerPage) { throw new RuntimeException("Stub!"); }

/**
 * Sets ranking strategy for AppSearch results.
 * @param rankingStrategy Value is {@link android.app.appsearch.SearchSpec#RANKING_STRATEGY_NONE}, {@link android.app.appsearch.SearchSpec#RANKING_STRATEGY_DOCUMENT_SCORE}, {@link android.app.appsearch.SearchSpec#RANKING_STRATEGY_CREATION_TIMESTAMP}, {@link android.app.appsearch.SearchSpec#RANKING_STRATEGY_RELEVANCE_SCORE}, {@link android.app.appsearch.SearchSpec#RANKING_STRATEGY_USAGE_COUNT}, {@link android.app.appsearch.SearchSpec#RANKING_STRATEGY_USAGE_LAST_USED_TIMESTAMP}, {@link android.app.appsearch.SearchSpec#RANKING_STRATEGY_SYSTEM_USAGE_COUNT}, {@link android.app.appsearch.SearchSpec#RANKING_STRATEGY_SYSTEM_USAGE_LAST_USED_TIMESTAMP}, {@link android.app.appsearch.SearchSpec#RANKING_STRATEGY_JOIN_AGGREGATE_SCORE}, or {@link android.app.appsearch.SearchSpec#RANKING_STRATEGY_ADVANCED_RANKING_EXPRESSION}
 * @return This value cannot be {@code null}.
 */

@android.annotation.NonNull
public android.app.appsearch.SearchSpec.Builder setRankingStrategy(int rankingStrategy) { throw new RuntimeException("Stub!"); }

/**
 * Enables advanced ranking to score based on {@code advancedRankingExpression}.
 *
 * <p>This method will set RankingStrategy to {@link
 * #RANKING_STRATEGY_ADVANCED_RANKING_EXPRESSION}.
 *
 * <p>The ranking expression is a mathematical expression that will be evaluated to a
 * floating-point number of double type representing the score of each document.
 *
 * <p>Numeric literals, arithmetic operators, mathematical functions, and document-based
 * functions are supported to build expressions.
 *
 * <p>The following are examples of numeric literals:
 *
 * <ul>
 *   <li>Integer
 *       <p>Example: 0, 1, 2, 13
 *   <li>Floating-point number
 *       <p>Example: 0.333, 0.5, 123.456
 *   <li>Negative number
 *       <p>Example: -5, -10.5, -100.123
 * </ul>
 *
 * <p>The following are supported arithmetic operators:
 *
 * <ul>
 *   <li>Addition(+)
 *       <p>Example: "1 + 1" will be evaluated to 2.
 *   <li>Subtraction(-)
 *       <p>Example: "2 - 1.5" will be evaluated to 0.5.
 *   <li>Multiplication(*)
 *       <p>Example: "2 * -2" will be evaluated to -4.
 *   <li>Division(/)
 *       <p>Example: "5 / 2" will be evaluated to 2.5.
 * </ul>
 *
 * <p>Multiplication and division have higher precedences than addition and subtraction, but
 * multiplication has the same precedence as division, and addition has the same precedence
 * as subtraction. Parentheses are supported to change precedences.
 *
 * <p>For example:
 *
 * <ul>
 *   <li>"2 + 3 - 4 * 5" will be evaluated to -15
 *   <li>"(2 + 3) - (4 * 5)" will be evaluated to -15
 *   <li>"2 + (3 - 4) * 5" will be evaluated to -3
 * </ul>
 *
 * <p>The following are supported mathematical functions:
 *
 * <ul>
 *   <li>log(x) - the natural log of x
 *   <li>log(x, y) - the log of y with base x
 *   <li>pow(x, y) - x to the power of y
 *   <li>max(v1, v2, ..., vn) with n > 0 - the maximum value among v1, ..., vn
 *   <li>min(v1, v2, ..., vn) with n > 0 - the minimum value among v1, ..., vn
 *   <li>sqrt(x) - the square root of x
 *   <li>abs(x) - the absolute value of x
 *   <li>sin(x), cos(x), tan(x) - trigonometric functions of x
 *   <li>Example: "max(abs(-100), 10) + pow(2, 10)" will be evaluated to 1124
 * </ul>
 *
 * <p>Document-based functions must be called via "this", which represents the current
 * document being scored. The following are supported document-based functions:
 *
 * <ul>
 *   <li>this.documentScore()
 *       <p>Get the app-provided document score of the current document. This is the same
 *       score that is returned for {@link #RANKING_STRATEGY_DOCUMENT_SCORE}.
 *   <li>this.creationTimestamp()
 *       <p>Get the creation timestamp of the current document. This is the same score that
 *       is returned for {@link #RANKING_STRATEGY_CREATION_TIMESTAMP}.
 *   <li>this.relevanceScore()
 *       <p>Get the BM25F relevance score of the current document in relation to the query
 *       string. This is the same score that is returned for {@link
 *       #RANKING_STRATEGY_RELEVANCE_SCORE}.
 *   <li>this.usageCount(type) and this.usageLastUsedTimestamp(type)
 *       <p>Get the number of usages or the timestamp of last usage by type for the current
 *       document, where type must be evaluated to an integer from 1 to 2. Type 1 refers to
 *       usages reported by {@link android.app.appsearch.AppSearchSession#reportUsage AppSearchSession#reportUsage}, and type 2 refers to
 *       usages reported by {@link android.app.appsearch.GlobalSearchSession#reportSystemUsage GlobalSearchSession#reportSystemUsage}.
 * </ul>
 *
 * <p>Some errors may occur when using advanced ranking.
 *
 * <ul>
 *   <li>Syntax Error: the expression violates the syntax of the advanced ranking language,
 *       such as unbalanced parenthesis.
 *   <li>Type Error: the expression fails a static type check, such as getting the wrong
 *       number of arguments for a function.
 *   <li>Evaluation Error: an error occurred while evaluating the value of the expression,
 *       such as getting a non-finite value in the middle of evaluation. Expressions like "1
 *       / 0" and "log(0) fall into this category.
 * </ul>
 *
 * <p>Syntax errors and type errors will fail the entire search and will cause {@link android.app.appsearch.SearchResults#getNextPage  } to throw an {@link android.app.appsearch.exceptions.AppSearchException AppSearchException}.
 *
 * <p>Evaluation errors will result in the offending documents receiving the default score.
 * For {@link #ORDER_DESCENDING}, the default score will be 0, for {@link #ORDER_ASCENDING}
 * the default score will be infinity.
 *
 * @param advancedRankingExpression a non-empty string representing the ranking expression.
 
 * This value cannot be {@code null}.
 * @return This value cannot be {@code null}.
 */

@android.annotation.NonNull
public android.app.appsearch.SearchSpec.Builder setRankingStrategy(@android.annotation.NonNull java.lang.String advancedRankingExpression) { throw new RuntimeException("Stub!"); }

/**
 * Indicates the order of returned search results, the default is {@link #ORDER_DESCENDING},
 * meaning that results with higher scores come first.
 *
 * <p>This order field will be ignored if RankingStrategy = {@code RANKING_STRATEGY_NONE}.
 
 * @param order Value is {@link android.app.appsearch.SearchSpec#ORDER_DESCENDING}, or {@link android.app.appsearch.SearchSpec#ORDER_ASCENDING}
 * @return This value cannot be {@code null}.
 */

@android.annotation.NonNull
public android.app.appsearch.SearchSpec.Builder setOrder(int order) { throw new RuntimeException("Stub!"); }

/**
 * Only the first {@code snippetCount} documents based on the ranking strategy will have
 * snippet information provided.
 *
 * <p>The list returned from {@link android.app.appsearch.SearchResult#getMatchInfos SearchResult#getMatchInfos} will contain at most this
 * many entries.
 *
 * <p>If set to 0 (default), snippeting is disabled and the list returned from {@link android.app.appsearch.SearchResult#getMatchInfos  } will be empty.
 
 * @param snippetCount Value is between 0 and MAX_SNIPPET_COUNT inclusive
 * @return This value cannot be {@code null}.
 */

@android.annotation.NonNull
public android.app.appsearch.SearchSpec.Builder setSnippetCount(int snippetCount) { throw new RuntimeException("Stub!"); }

/**
 * Sets {@code snippetCountPerProperty}. Only the first {@code snippetCountPerProperty}
 * snippets for each property of each {@link android.app.appsearch.GenericDocument GenericDocument} will contain snippet
 * information.
 *
 * <p>If set to 0, snippeting is disabled and the list returned from {@link android.app.appsearch.SearchResult#getMatchInfos  } will be empty.
 *
 * <p>The default behavior is to snippet all matches a property contains, up to the maximum
 * value of 10,000.
 
 * @param snippetCountPerProperty Value is between 0 and MAX_SNIPPET_PER_PROPERTY_COUNT inclusive
 * @return This value cannot be {@code null}.
 */

@android.annotation.NonNull
public android.app.appsearch.SearchSpec.Builder setSnippetCountPerProperty(int snippetCountPerProperty) { throw new RuntimeException("Stub!"); }

/**
 * Sets {@code maxSnippetSize}, the maximum snippet size. Snippet windows start at {@code
 * maxSnippetSize/2} bytes before the middle of the matching token and end at {@code
 * maxSnippetSize/2} bytes after the middle of the matching token. It respects token
 * boundaries, therefore the returned window may be smaller than requested.
 *
 * <p>Setting {@code maxSnippetSize} to 0 will disable windowing and an empty string will be
 * returned. If matches enabled is also set to false, then snippeting is disabled.
 *
 * <p>Ex. {@code maxSnippetSize} = 16. "foo bar baz bat rat" with a query of "baz" will
 * return a window of "bar baz bat" which is only 11 bytes long.
 
 * @param maxSnippetSize Value is between 0 and MAX_SNIPPET_SIZE_LIMIT inclusive
 * @return This value cannot be {@code null}.
 */

@android.annotation.NonNull
public android.app.appsearch.SearchSpec.Builder setMaxSnippetSize(int maxSnippetSize) { throw new RuntimeException("Stub!"); }

/**
 * Adds property paths for the specified type to be used for projection. If property paths
 * are added for a type, then only the properties referred to will be retrieved for results
 * of that type. If a property path that is specified isn't present in a result, it will be
 * ignored for that result. Property paths cannot be null.
 *
 * @see #addProjectionPaths
 * @param schema a string corresponding to the schema to add projections to.
 * This value cannot be {@code null}.
 * @param propertyPaths the projections to add.

 * This value cannot be {@code null}.
 */

@android.annotation.NonNull
public android.app.appsearch.SearchSpec.Builder addProjection(@android.annotation.NonNull java.lang.String schema, @android.annotation.NonNull java.util.Collection<java.lang.String> propertyPaths) { throw new RuntimeException("Stub!"); }

/**
 * Adds property paths for the specified type to be used for projection. If property paths
 * are added for a type, then only the properties referred to will be retrieved for results
 * of that type. If a property path that is specified isn't present in a result, it will be
 * ignored for that result. Property paths cannot be null.
 *
 * <p>If no property paths are added for a particular type, then all properties of results
 * of that type will be retrieved.
 *
 * <p>If property path is added for the {@link android.app.appsearch.SearchSpec#PROJECTION_SCHEMA_TYPE_WILDCARD SearchSpec#PROJECTION_SCHEMA_TYPE_WILDCARD},
 * then those property paths will apply to all results, excepting any types that have their
 * own, specific property paths set.
 *
 * <p>Suppose the following document is in the index.
 *
 * <pre>{@code
 * Email: Document {
 *   sender: Document {
 *     name: "Mr. Person"
 *     email: "mrperson123@google.com"
 *   }
 *   recipients: [
 *     Document {
 *       name: "John Doe"
 *       email: "johndoe123@google.com"
 *     }
 *     Document {
 *       name: "Jane Doe"
 *       email: "janedoe123@google.com"
 *     }
 *   ]
 *   subject: "IMPORTANT"
 *   body: "Limited time offer!"
 * }
 * }</pre>
 *
 * <p>Then, suppose that a query for "important" is issued with the following projection
 * type property paths:
 *
 * <pre>{@code
 * {schema: "Email", ["subject", "sender.name", "recipients.name"]}
 * }</pre>
 *
 * <p>The above document will be returned as:
 *
 * <pre>{@code
 * Email: Document {
 *   sender: Document {
 *     name: "Mr. Body"
 *   }
 *   recipients: [
 *     Document {
 *       name: "John Doe"
 *     }
 *     Document {
 *       name: "Jane Doe"
 *     }
 *   ]
 *   subject: "IMPORTANT"
 * }
 * }</pre>
 *
 * @param schema a string corresponding to the schema to add projections to.
 * This value cannot be {@code null}.
 * @param propertyPaths the projections to add.

 * This value cannot be {@code null}.
 */

@android.annotation.NonNull
public android.app.appsearch.SearchSpec.Builder addProjectionPaths(@android.annotation.NonNull java.lang.String schema, @android.annotation.NonNull java.util.Collection<android.app.appsearch.PropertyPath> propertyPaths) { throw new RuntimeException("Stub!"); }

/**
 * Sets the maximum number of results to return for each group, where groups are defined by
 * grouping type.
 *
 * <p>Calling this method will override any previous calls. So calling
 * setResultGrouping(GROUPING_TYPE_PER_PACKAGE, 7) and then calling
 * setResultGrouping(GROUPING_TYPE_PER_PACKAGE, 2) will result in only the latter, a limit
 * of two results per package, being applied. Or calling setResultGrouping
 * (GROUPING_TYPE_PER_PACKAGE, 1) and then calling setResultGrouping
 * (GROUPING_TYPE_PER_PACKAGE | GROUPING_PER_NAMESPACE, 5) will result in five results per
 * package per namespace.
 *
 * @param groupingTypeFlags One or more combination of grouping types.
 * Value is either <code>0</code> or a combination of {@link android.app.appsearch.SearchSpec#GROUPING_TYPE_PER_PACKAGE}, {@link android.app.appsearch.SearchSpec#GROUPING_TYPE_PER_NAMESPACE}, and android.app.appsearch.SearchSpec.GROUPING_TYPE_PER_SCHEMA
 * @param limit Number of results to return per {@code groupingTypeFlags}.
 * @throws java.lang.IllegalArgumentException if groupingTypeFlags is zero.
 
 * @return This value cannot be {@code null}.
 */

@android.annotation.NonNull
public android.app.appsearch.SearchSpec.Builder setResultGrouping(int groupingTypeFlags, int limit) { throw new RuntimeException("Stub!"); }

/**
 * Sets property weights by schema type and property path.
 *
 * <p>Property weights are used to promote and demote query term matches within a {@link android.app.appsearch.GenericDocument GenericDocument} property when applying scoring.
 *
 * <p>Property weights must be positive values (greater than 0). A property's weight is
 * multiplied with that property's scoring contribution. This means weights set between 0.0
 * and 1.0 demote scoring contributions by a term match within the property. Weights set
 * above 1.0 promote scoring contributions by a term match within the property.
 *
 * <p>Properties that exist in the {@link android.app.appsearch.AppSearchSchema AppSearchSchema}, but do not have a weight
 * explicitly set will be given a default weight of 1.0.
 *
 * <p>Weights set for property paths that do not exist in the {@link android.app.appsearch.AppSearchSchema AppSearchSchema} will
 * be discarded and not affect scoring.
 *
 * <p><b>NOTE:</b> Property weights only affect scoring for query-dependent scoring
 * strategies, such as {@link #RANKING_STRATEGY_RELEVANCE_SCORE}.
 *
 * @param schemaType the schema type to set property weights for.
 * This value cannot be {@code null}.
 * @param propertyPathWeights a {@link java.util.Map Map} of property paths of the schema type to the
 *     weight to set for that property.
 * This value cannot be {@code null}.
 * @throws java.lang.IllegalArgumentException if a weight is equal to or less than 0.0.
 
 * @return This value cannot be {@code null}.
 */

@android.annotation.NonNull
public android.app.appsearch.SearchSpec.Builder setPropertyWeights(@android.annotation.NonNull java.lang.String schemaType, @android.annotation.NonNull java.util.Map<java.lang.String,java.lang.Double> propertyPathWeights) { throw new RuntimeException("Stub!"); }

/**
 * Specifies which documents to join with, and how to join.
 *
 * <p>If the ranking strategy is {@link #RANKING_STRATEGY_JOIN_AGGREGATE_SCORE}, and the
 * JoinSpec is null, {@link #build} will throw an {@link android.app.appsearch.exceptions.AppSearchException AppSearchException}.
 *
 * @param joinSpec a specification on how to perform the Join operation.

 * This value cannot be {@code null}.
 */

@android.annotation.NonNull
public android.app.appsearch.SearchSpec.Builder setJoinSpec(@android.annotation.NonNull android.app.appsearch.JoinSpec joinSpec) { throw new RuntimeException("Stub!"); }

/**
 * Sets property weights by schema type and property path.
 *
 * <p>Property weights are used to promote and demote query term matches within a {@link android.app.appsearch.GenericDocument GenericDocument} property when applying scoring.
 *
 * <p>Property weights must be positive values (greater than 0). A property's weight is
 * multiplied with that property's scoring contribution. This means weights set between 0.0
 * and 1.0 demote scoring contributions by a term match within the property. Weights set
 * above 1.0 promote scoring contributions by a term match within the property.
 *
 * <p>Properties that exist in the {@link android.app.appsearch.AppSearchSchema AppSearchSchema}, but do not have a weight
 * explicitly set will be given a default weight of 1.0.
 *
 * <p>Weights set for property paths that do not exist in the {@link android.app.appsearch.AppSearchSchema AppSearchSchema} will
 * be discarded and not affect scoring.
 *
 * <p><b>NOTE:</b> Property weights only affect scoring for query-dependent scoring
 * strategies, such as {@link #RANKING_STRATEGY_RELEVANCE_SCORE}.
 *
 * @param schemaType the schema type to set property weights for.
 * This value cannot be {@code null}.
 * @param propertyPathWeights a {@link java.util.Map Map} of property paths of the schema type to the
 *     weight to set for that property.
 * This value cannot be {@code null}.
 * @throws java.lang.IllegalArgumentException if a weight is equal to or less than 0.0.
 
 * @return This value cannot be {@code null}.
 */

@android.annotation.NonNull
public android.app.appsearch.SearchSpec.Builder setPropertyWeightPaths(@android.annotation.NonNull java.lang.String schemaType, @android.annotation.NonNull java.util.Map<android.app.appsearch.PropertyPath,java.lang.Double> propertyPathWeights) { throw new RuntimeException("Stub!"); }

/**
 * Sets the {@link Features#NUMERIC_SEARCH} feature as enabled/disabled according to the
 * enabled parameter.
 *
 * @param enabled Enables the feature if true, otherwise disables it.
 *     <p>If disabled, disallows use of {@link android.app.appsearch.AppSearchSchema.LongPropertyConfig#INDEXING_TYPE_RANGE      } and all other numeric
 *     querying features.

 * @return This value cannot be {@code null}.
 */

@android.annotation.NonNull
public android.app.appsearch.SearchSpec.Builder setNumericSearchEnabled(boolean enabled) { throw new RuntimeException("Stub!"); }

/**
 * Sets the {@link Features#VERBATIM_SEARCH} feature as enabled/disabled according to the
 * enabled parameter.
 *
 * @param enabled Enables the feature if true, otherwise disables it
 *     <p>If disabled, disallows use of {@link android.app.appsearch.AppSearchSchema.StringPropertyConfig#TOKENIZER_TYPE_VERBATIM      } and all other verbatim
 *     search features within the query language that allows clients to search using the
 *     verbatim string operator.
 *     <p>Ex. The verbatim string operator '"foo/bar" OR baz' will ensure that 'foo/bar' is
 *     treated as a single 'verbatim' token.

 * @return This value cannot be {@code null}.
 */

@android.annotation.NonNull
public android.app.appsearch.SearchSpec.Builder setVerbatimSearchEnabled(boolean enabled) { throw new RuntimeException("Stub!"); }

/**
 * Sets the {@link Features#LIST_FILTER_QUERY_LANGUAGE} feature as enabled/disabled
 * according to the enabled parameter.
 *
 * @param enabled Enables the feature if true, otherwise disables it.
 *     <p>This feature covers the expansion of the query language to conform to the
 *     definition of the list filters language (https://aip.dev/160). This includes:
 *     <ul>
 *       <li>addition of explicit 'AND' and 'NOT' operators
 *       <li>property restricts are allowed with grouping (ex. "prop:(a OR b)")
 *       <li>addition of custom functions to control matching
 *     </ul>
 *     <p>The newly added custom functions covered by this feature are:
 *     <ul>
 *       <li>createList(String...)
 *       <li>termSearch(String, List<String>)
 *     </ul>
 *     <p>createList takes a variable number of strings and returns a list of strings. It is
 *     for use with termSearch.
 *     <p>termSearch takes a query string that will be parsed according to the supported
 *     query language and an optional list of strings that specify the properties to be
 *     restricted to. This exists as a convenience for multiple property restricts. So, for
 *     example, the query "(subject:foo OR body:foo) (subject:bar OR body:bar)" could be
 *     rewritten as "termSearch(\"foo bar\", createList(\"subject\", \"bar\"))"

 * @return This value cannot be {@code null}.
 */

@android.annotation.NonNull
public android.app.appsearch.SearchSpec.Builder setListFilterQueryLanguageEnabled(boolean enabled) { throw new RuntimeException("Stub!"); }

/**
 * Constructs a new {@link android.app.appsearch.SearchSpec SearchSpec} from the contents of this builder.
 *
 * @throws java.lang.IllegalArgumentException if property weights are provided with a ranking strategy
 *     that isn't RANKING_STRATEGY_RELEVANCE_SCORE.
 * @throws java.lang.IllegalStateException if the ranking strategy is {@link
 *     #RANKING_STRATEGY_JOIN_AGGREGATE_SCORE} and {@link #setJoinSpec} has never been
 *     called.
 * @throws java.lang.IllegalStateException if the aggregation scoring strategy has been set in {@link android.app.appsearch.JoinSpec#getAggregationScoringStrategy()      } but the ranking strategy is not {@link
 *     #RANKING_STRATEGY_JOIN_AGGREGATE_SCORE}.

 * @return This value cannot be {@code null}.
 */

@android.annotation.NonNull
public android.app.appsearch.SearchSpec build() { throw new RuntimeException("Stub!"); }
}

}

