/*
 * Copyright (C) 2014 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */


package android.net;

import java.util.Set;

/**
 * An object describing a network that the application is interested in.
 *
 * <p>@see <a href="https://developer.android.com/training/basics/network-ops/reading-network-state>
 * this general guide</a> on how to use NetworkCapabilities and related classes.
 *
 * NetworkRequest defines a request for a network, made through
 * {@link android.net.NetworkRequest.Builder NetworkRequest.Builder} and used to request a network via
 * {@link android.net.ConnectivityManager#requestNetwork ConnectivityManager#requestNetwork} or to listen for changes
 * via the {@link android.net.ConnectivityManager#registerNetworkCallback ConnectivityManager#registerNetworkCallback} family of
 * functions.
 *
 * <p>{@link android.net.ConnectivityManager#requestNetwork ConnectivityManager#requestNetwork} will try to find a connected
 * network matching the NetworkRequest, and return it if there is one.
 * As long as the request is outstanding, the system will try to find the best
 * possible network that matches the request. The request will keep up the
 * currently best connected network, and if a better one is found (e.g. cheaper
 * or faster) the system will bring up that better network to better serve the
 * request. A request filed with {@link android.net.ConnectivityManager#requestNetwork ConnectivityManager#requestNetwork} will
 * only match one network at a time (the one the system thinks is best), even if
 * other networks can match the request that are being kept up by other requests.
 *
 * For example, an application needing a network with
 * {@link android.net.NetworkCapabilities#NET_CAPABILITY_INTERNET NetworkCapabilities#NET_CAPABILITY_INTERNET} should use
 * {@link android.net.ConnectivityManager#requestNetwork ConnectivityManager#requestNetwork} to request the system keeps one up.
 * A general cellular network can satisfy this request, but if the system finds
 * a free Wi-Fi network which is expected to be faster, it will try and connect
 * to that Wi-Fi network and switch the request over to it once it is connected.
 * The cell network may stay connected if there are outstanding requests (from
 * the same app or from other apps on the system) that match the cell network
 * but not the Wi-Fi network, such as a request with {@link android.net.NetworkCapabilities#NET_CAPABILITY_MMS NetworkCapabilities#NET_CAPABILITY_MMS}.
 *
 * When a network is no longer needed to serve any request, the system can
 * tear it down at any time and usually does so immediately, so make sure to
 * keep up requests for the networks your app needs.
 *
 * <p>By contrast, requests filed with {@link android.net.ConnectivityManager#registerNetworkCallback ConnectivityManager#registerNetworkCallback}
 * will receive callbacks for all matching networks, and will not cause the system to
 * keep up the networks they match. Use this to listen to networks that the device is
 * connected to, but that you don't want the system to keep up for your use case.
 *
 * <p>Applications build a NetworkRequest and pass it to one of the
 * {@link android.net.ConnectivityManager ConnectivityManager} methods above together with a
 * {@link android.net.ConnectivityManager.NetworkCallback ConnectivityManager.NetworkCallback} object. The callback
 * will then start receiving method calls about networks that match
 * the request.
 *
 * <p>Networks are brought up and/or matched according to the capabilities
 * set in the builder. For example, a request with
 * {@link android.net.NetworkCapabilities#NET_CAPABILITY_MMS NetworkCapabilities#NET_CAPABILITY_MMS} lets the system match
 * and/or bring up a network that is capable to send MMS. A request with
 * {@link android.net.NetworkCapabilities#NET_CAPABILITY_NOT_METERED NetworkCapabilities#NET_CAPABILITY_NOT_METERED} matches a network
 * that doesn't charge the user for usage. See
 * {@link android.net.NetworkCapabilities NetworkCapabilities} for a list of capabilities and their
 * description.
 *
 * <p>While all capabilities can be matched with the
 * {@link android.net.ConnectivityManager#registerNetworkCallback ConnectivityManager#registerNetworkCallback} family of methods,
 * not all capabilities can be used to request that the system brings
 * up a network with {@link android.net.ConnectivityManager#requestNetwork ConnectivityManager#requestNetwork}. For example,
 * an application cannot use {@link android.net.ConnectivityManager#requestNetwork ConnectivityManager#requestNetwork} to
 * ask the system to bring up a network with
 * {@link android.net.NetworkCapabilities#NET_CAPABILITY_CAPTIVE_PORTAL NetworkCapabilities#NET_CAPABILITY_CAPTIVE_PORTAL}, because the
 * system won't know if a network has a captive portal before it connects
 * to that network. Similarly, some capabilities may require a specific
 * permission or privilege to be requested.
 *
 * Look up the specific capability and the {@link android.net.ConnectivityManager#requestNetwork ConnectivityManager#requestNetwork}
 * method for limitations applicable to each capability.
 *
 * <p>Also, starting with {@link android.os.Build.VERSION_CODES#UPSIDE_DOWN_CAKE Build.VERSION_CODES#UPSIDE_DOWN_CAKE}, some capabilities
 * require the application to self-certify by explicitly adding the
 * {@link android.content.pm.PackageManager#PROPERTY_SELF_CERTIFIED_NETWORK_CAPABILITIES}
 * property in the AndroidManifest.xml, which points to an XML resource file. In the
 * XML resource file, the application declares what kind of network capabilities the application
 * wants to have.
 *
 * Here is an example self-certification XML resource file :
 * <pre>
 *  {@code
 *  <network-capabilities-declaration xmlns:android="http://schemas.android.com/apk/res/android">
 *     <uses-network-capability android:name="NET_CAPABILITY_PRIORITIZE_LATENCY"/>
 *     <uses-network-capability android:name="NET_CAPABILITY_PRIORITIZE_BANDWIDTH"/>
 * </network-capabilities-declaration>
 *  }
 *  </pre>
 * Look up the specific capability to learn whether its usage requires this self-certification.
 */

@SuppressWarnings({"unchecked", "deprecation", "all"})
public class NetworkRequest implements android.os.Parcelable {

NetworkRequest() { throw new RuntimeException("Stub!"); }

public int describeContents() { throw new RuntimeException("Stub!"); }

public void writeToParcel(android.os.Parcel dest, int flags) { throw new RuntimeException("Stub!"); }

/**
 * @see android.net.NetworkRequest.Builder#addCapability(int)

 * @param capability Value is {@link android.net.NetworkCapabilities#NET_CAPABILITY_MMS}, {@link android.net.NetworkCapabilities#NET_CAPABILITY_SUPL}, {@link android.net.NetworkCapabilities#NET_CAPABILITY_DUN}, {@link android.net.NetworkCapabilities#NET_CAPABILITY_FOTA}, {@link android.net.NetworkCapabilities#NET_CAPABILITY_IMS}, {@link android.net.NetworkCapabilities#NET_CAPABILITY_CBS}, {@link android.net.NetworkCapabilities#NET_CAPABILITY_WIFI_P2P}, {@link android.net.NetworkCapabilities#NET_CAPABILITY_IA}, {@link android.net.NetworkCapabilities#NET_CAPABILITY_RCS}, {@link android.net.NetworkCapabilities#NET_CAPABILITY_XCAP}, {@link android.net.NetworkCapabilities#NET_CAPABILITY_EIMS}, {@link android.net.NetworkCapabilities#NET_CAPABILITY_NOT_METERED}, {@link android.net.NetworkCapabilities#NET_CAPABILITY_INTERNET}, {@link android.net.NetworkCapabilities#NET_CAPABILITY_NOT_RESTRICTED}, {@link android.net.NetworkCapabilities#NET_CAPABILITY_TRUSTED}, {@link android.net.NetworkCapabilities#NET_CAPABILITY_NOT_VPN}, {@link android.net.NetworkCapabilities#NET_CAPABILITY_VALIDATED}, {@link android.net.NetworkCapabilities#NET_CAPABILITY_CAPTIVE_PORTAL}, {@link android.net.NetworkCapabilities#NET_CAPABILITY_NOT_ROAMING}, {@link android.net.NetworkCapabilities#NET_CAPABILITY_FOREGROUND}, {@link android.net.NetworkCapabilities#NET_CAPABILITY_NOT_CONGESTED}, {@link android.net.NetworkCapabilities#NET_CAPABILITY_NOT_SUSPENDED}, android.net.NetworkCapabilities.NET_CAPABILITY_OEM_PAID, {@link android.net.NetworkCapabilities#NET_CAPABILITY_MCX}, android.net.NetworkCapabilities.NET_CAPABILITY_PARTIAL_CONNECTIVITY, {@link android.net.NetworkCapabilities#NET_CAPABILITY_TEMPORARILY_NOT_METERED}, android.net.NetworkCapabilities.NET_CAPABILITY_OEM_PRIVATE, android.net.NetworkCapabilities.NET_CAPABILITY_VEHICLE_INTERNAL, android.net.NetworkCapabilities.NET_CAPABILITY_NOT_VCN_MANAGED, {@link android.net.NetworkCapabilities#NET_CAPABILITY_ENTERPRISE}, android.net.NetworkCapabilities.NET_CAPABILITY_VSIM, android.net.NetworkCapabilities.NET_CAPABILITY_BIP, {@link android.net.NetworkCapabilities#NET_CAPABILITY_HEAD_UNIT}, {@link android.net.NetworkCapabilities#NET_CAPABILITY_MMTEL}, {@link android.net.NetworkCapabilities#NET_CAPABILITY_PRIORITIZE_LATENCY}, {@link android.net.NetworkCapabilities#NET_CAPABILITY_PRIORITIZE_BANDWIDTH}, {@link android.net.NetworkCapabilities#NET_CAPABILITY_LOCAL_NETWORK}, or android.net.NetworkCapabilities.NET_CAPABILITY_NOT_BANDWIDTH_CONSTRAINED
 */

public boolean hasCapability(int capability) { throw new RuntimeException("Stub!"); }

/**
 * Returns true if and only if the capabilities requested in this NetworkRequest are satisfied
 * by the provided {@link android.net.NetworkCapabilities NetworkCapabilities}.
 *
 * @param nc Capabilities that should satisfy this NetworkRequest. null capabilities do not
 *           satisfy any request.
 */

public boolean canBeSatisfiedBy(@android.annotation.Nullable android.net.NetworkCapabilities nc) { throw new RuntimeException("Stub!"); }

/**
 * @see android.net.NetworkRequest.Builder#addTransportType(int)

 * @param transportType Value is {@link android.net.NetworkCapabilities#TRANSPORT_CELLULAR}, {@link android.net.NetworkCapabilities#TRANSPORT_WIFI}, {@link android.net.NetworkCapabilities#TRANSPORT_BLUETOOTH}, {@link android.net.NetworkCapabilities#TRANSPORT_ETHERNET}, {@link android.net.NetworkCapabilities#TRANSPORT_VPN}, {@link android.net.NetworkCapabilities#TRANSPORT_WIFI_AWARE}, {@link android.net.NetworkCapabilities#TRANSPORT_LOWPAN}, android.net.NetworkCapabilities.TRANSPORT_TEST, {@link android.net.NetworkCapabilities#TRANSPORT_USB}, {@link android.net.NetworkCapabilities#TRANSPORT_THREAD}, or {@link android.net.NetworkCapabilities#TRANSPORT_SATELLITE}
 */

public boolean hasTransport(int transportType) { throw new RuntimeException("Stub!"); }

/**
 * @see android.net.NetworkRequest.Builder#setNetworkSpecifier(NetworkSpecifier)

 * @return This value may be {@code null}.
 */

@android.annotation.Nullable
public android.net.NetworkSpecifier getNetworkSpecifier() { throw new RuntimeException("Stub!"); }

public java.lang.String toString() { throw new RuntimeException("Stub!"); }

/**
 * {@inheritDoc}

 * @param obj This value may be {@code null}.
 */

public boolean equals(@android.annotation.Nullable java.lang.Object obj) { throw new RuntimeException("Stub!"); }

public int hashCode() { throw new RuntimeException("Stub!"); }

/**
 * Gets all the capabilities set on this {@code NetworkRequest} instance.
 *
 * @return an array of capability values for this instance.
 
 * This value cannot be {@code null}.

 * Value is {@link android.net.NetworkCapabilities#NET_CAPABILITY_MMS}, {@link android.net.NetworkCapabilities#NET_CAPABILITY_SUPL}, {@link android.net.NetworkCapabilities#NET_CAPABILITY_DUN}, {@link android.net.NetworkCapabilities#NET_CAPABILITY_FOTA}, {@link android.net.NetworkCapabilities#NET_CAPABILITY_IMS}, {@link android.net.NetworkCapabilities#NET_CAPABILITY_CBS}, {@link android.net.NetworkCapabilities#NET_CAPABILITY_WIFI_P2P}, {@link android.net.NetworkCapabilities#NET_CAPABILITY_IA}, {@link android.net.NetworkCapabilities#NET_CAPABILITY_RCS}, {@link android.net.NetworkCapabilities#NET_CAPABILITY_XCAP}, {@link android.net.NetworkCapabilities#NET_CAPABILITY_EIMS}, {@link android.net.NetworkCapabilities#NET_CAPABILITY_NOT_METERED}, {@link android.net.NetworkCapabilities#NET_CAPABILITY_INTERNET}, {@link android.net.NetworkCapabilities#NET_CAPABILITY_NOT_RESTRICTED}, {@link android.net.NetworkCapabilities#NET_CAPABILITY_TRUSTED}, {@link android.net.NetworkCapabilities#NET_CAPABILITY_NOT_VPN}, {@link android.net.NetworkCapabilities#NET_CAPABILITY_VALIDATED}, {@link android.net.NetworkCapabilities#NET_CAPABILITY_CAPTIVE_PORTAL}, {@link android.net.NetworkCapabilities#NET_CAPABILITY_NOT_ROAMING}, {@link android.net.NetworkCapabilities#NET_CAPABILITY_FOREGROUND}, {@link android.net.NetworkCapabilities#NET_CAPABILITY_NOT_CONGESTED}, {@link android.net.NetworkCapabilities#NET_CAPABILITY_NOT_SUSPENDED}, android.net.NetworkCapabilities.NET_CAPABILITY_OEM_PAID, {@link android.net.NetworkCapabilities#NET_CAPABILITY_MCX}, android.net.NetworkCapabilities.NET_CAPABILITY_PARTIAL_CONNECTIVITY, {@link android.net.NetworkCapabilities#NET_CAPABILITY_TEMPORARILY_NOT_METERED}, android.net.NetworkCapabilities.NET_CAPABILITY_OEM_PRIVATE, android.net.NetworkCapabilities.NET_CAPABILITY_VEHICLE_INTERNAL, android.net.NetworkCapabilities.NET_CAPABILITY_NOT_VCN_MANAGED, {@link android.net.NetworkCapabilities#NET_CAPABILITY_ENTERPRISE}, android.net.NetworkCapabilities.NET_CAPABILITY_VSIM, android.net.NetworkCapabilities.NET_CAPABILITY_BIP, {@link android.net.NetworkCapabilities#NET_CAPABILITY_HEAD_UNIT}, {@link android.net.NetworkCapabilities#NET_CAPABILITY_MMTEL}, {@link android.net.NetworkCapabilities#NET_CAPABILITY_PRIORITIZE_LATENCY}, {@link android.net.NetworkCapabilities#NET_CAPABILITY_PRIORITIZE_BANDWIDTH}, {@link android.net.NetworkCapabilities#NET_CAPABILITY_LOCAL_NETWORK}, or android.net.NetworkCapabilities.NET_CAPABILITY_NOT_BANDWIDTH_CONSTRAINED
 */

@android.annotation.NonNull
public int[] getCapabilities() { throw new RuntimeException("Stub!"); }

/**
 * Gets all the transports set on this {@code NetworkRequest} instance.
 *
 * @return an array of transport type values for this instance.
 
 * This value cannot be {@code null}.

 * Value is {@link android.net.NetworkCapabilities#TRANSPORT_CELLULAR}, {@link android.net.NetworkCapabilities#TRANSPORT_WIFI}, {@link android.net.NetworkCapabilities#TRANSPORT_BLUETOOTH}, {@link android.net.NetworkCapabilities#TRANSPORT_ETHERNET}, {@link android.net.NetworkCapabilities#TRANSPORT_VPN}, {@link android.net.NetworkCapabilities#TRANSPORT_WIFI_AWARE}, {@link android.net.NetworkCapabilities#TRANSPORT_LOWPAN}, android.net.NetworkCapabilities.TRANSPORT_TEST, {@link android.net.NetworkCapabilities#TRANSPORT_USB}, {@link android.net.NetworkCapabilities#TRANSPORT_THREAD}, or {@link android.net.NetworkCapabilities#TRANSPORT_SATELLITE}
 */

@android.annotation.NonNull
public int[] getTransportTypes() { throw new RuntimeException("Stub!"); }

/**
 * Gets all the subscription ids set on this {@code NetworkRequest} instance.
 *
 * @return Set of Integer values for this instance.

 * This value cannot be {@code null}.
 */

@android.annotation.NonNull
public java.util.Set<java.lang.Integer> getSubscriptionIds() { throw new RuntimeException("Stub!"); }

@android.annotation.NonNull public static final android.os.Parcelable.Creator<android.net.NetworkRequest> CREATOR;
static { CREATOR = null; }
/**
 * Builder used to create {@link android.net.NetworkRequest NetworkRequest} objects.  Specify the Network features
 * needed in terms of {@link android.net.NetworkCapabilities NetworkCapabilities} features
 */

@SuppressWarnings({"unchecked", "deprecation", "all"})
public static class Builder {

/**
 * Default constructor for Builder.
 */

public Builder() { throw new RuntimeException("Stub!"); }

/**
 * Creates a new Builder of NetworkRequest from an existing instance.

 * @param request This value cannot be {@code null}.
 */

public Builder(@android.annotation.NonNull android.net.NetworkRequest request) { throw new RuntimeException("Stub!"); }

/**
 * Build {@link android.net.NetworkRequest NetworkRequest} give the current set of capabilities.
 */

public android.net.NetworkRequest build() { throw new RuntimeException("Stub!"); }

/**
 * Add the given capability requirement to this builder.  These represent
 * the requested network's required capabilities.  Note that when searching
 * for a network to satisfy a request, all capabilities requested must be
 * satisfied.
 *
 * @param capability The capability to add.
 * Value is {@link android.net.NetworkCapabilities#NET_CAPABILITY_MMS}, {@link android.net.NetworkCapabilities#NET_CAPABILITY_SUPL}, {@link android.net.NetworkCapabilities#NET_CAPABILITY_DUN}, {@link android.net.NetworkCapabilities#NET_CAPABILITY_FOTA}, {@link android.net.NetworkCapabilities#NET_CAPABILITY_IMS}, {@link android.net.NetworkCapabilities#NET_CAPABILITY_CBS}, {@link android.net.NetworkCapabilities#NET_CAPABILITY_WIFI_P2P}, {@link android.net.NetworkCapabilities#NET_CAPABILITY_IA}, {@link android.net.NetworkCapabilities#NET_CAPABILITY_RCS}, {@link android.net.NetworkCapabilities#NET_CAPABILITY_XCAP}, {@link android.net.NetworkCapabilities#NET_CAPABILITY_EIMS}, {@link android.net.NetworkCapabilities#NET_CAPABILITY_NOT_METERED}, {@link android.net.NetworkCapabilities#NET_CAPABILITY_INTERNET}, {@link android.net.NetworkCapabilities#NET_CAPABILITY_NOT_RESTRICTED}, {@link android.net.NetworkCapabilities#NET_CAPABILITY_TRUSTED}, {@link android.net.NetworkCapabilities#NET_CAPABILITY_NOT_VPN}, {@link android.net.NetworkCapabilities#NET_CAPABILITY_VALIDATED}, {@link android.net.NetworkCapabilities#NET_CAPABILITY_CAPTIVE_PORTAL}, {@link android.net.NetworkCapabilities#NET_CAPABILITY_NOT_ROAMING}, {@link android.net.NetworkCapabilities#NET_CAPABILITY_FOREGROUND}, {@link android.net.NetworkCapabilities#NET_CAPABILITY_NOT_CONGESTED}, {@link android.net.NetworkCapabilities#NET_CAPABILITY_NOT_SUSPENDED}, android.net.NetworkCapabilities.NET_CAPABILITY_OEM_PAID, {@link android.net.NetworkCapabilities#NET_CAPABILITY_MCX}, android.net.NetworkCapabilities.NET_CAPABILITY_PARTIAL_CONNECTIVITY, {@link android.net.NetworkCapabilities#NET_CAPABILITY_TEMPORARILY_NOT_METERED}, android.net.NetworkCapabilities.NET_CAPABILITY_OEM_PRIVATE, android.net.NetworkCapabilities.NET_CAPABILITY_VEHICLE_INTERNAL, android.net.NetworkCapabilities.NET_CAPABILITY_NOT_VCN_MANAGED, {@link android.net.NetworkCapabilities#NET_CAPABILITY_ENTERPRISE}, android.net.NetworkCapabilities.NET_CAPABILITY_VSIM, android.net.NetworkCapabilities.NET_CAPABILITY_BIP, {@link android.net.NetworkCapabilities#NET_CAPABILITY_HEAD_UNIT}, {@link android.net.NetworkCapabilities#NET_CAPABILITY_MMTEL}, {@link android.net.NetworkCapabilities#NET_CAPABILITY_PRIORITIZE_LATENCY}, {@link android.net.NetworkCapabilities#NET_CAPABILITY_PRIORITIZE_BANDWIDTH}, {@link android.net.NetworkCapabilities#NET_CAPABILITY_LOCAL_NETWORK}, or android.net.NetworkCapabilities.NET_CAPABILITY_NOT_BANDWIDTH_CONSTRAINED
 * @return The builder to facilitate chaining
 *         {@code builder.addCapability(...).addCapability();}.
 */

public android.net.NetworkRequest.Builder addCapability(int capability) { throw new RuntimeException("Stub!"); }

/**
 * Removes (if found) the given capability from this builder instance.
 *
 * @param capability The capability to remove.
 * Value is {@link android.net.NetworkCapabilities#NET_CAPABILITY_MMS}, {@link android.net.NetworkCapabilities#NET_CAPABILITY_SUPL}, {@link android.net.NetworkCapabilities#NET_CAPABILITY_DUN}, {@link android.net.NetworkCapabilities#NET_CAPABILITY_FOTA}, {@link android.net.NetworkCapabilities#NET_CAPABILITY_IMS}, {@link android.net.NetworkCapabilities#NET_CAPABILITY_CBS}, {@link android.net.NetworkCapabilities#NET_CAPABILITY_WIFI_P2P}, {@link android.net.NetworkCapabilities#NET_CAPABILITY_IA}, {@link android.net.NetworkCapabilities#NET_CAPABILITY_RCS}, {@link android.net.NetworkCapabilities#NET_CAPABILITY_XCAP}, {@link android.net.NetworkCapabilities#NET_CAPABILITY_EIMS}, {@link android.net.NetworkCapabilities#NET_CAPABILITY_NOT_METERED}, {@link android.net.NetworkCapabilities#NET_CAPABILITY_INTERNET}, {@link android.net.NetworkCapabilities#NET_CAPABILITY_NOT_RESTRICTED}, {@link android.net.NetworkCapabilities#NET_CAPABILITY_TRUSTED}, {@link android.net.NetworkCapabilities#NET_CAPABILITY_NOT_VPN}, {@link android.net.NetworkCapabilities#NET_CAPABILITY_VALIDATED}, {@link android.net.NetworkCapabilities#NET_CAPABILITY_CAPTIVE_PORTAL}, {@link android.net.NetworkCapabilities#NET_CAPABILITY_NOT_ROAMING}, {@link android.net.NetworkCapabilities#NET_CAPABILITY_FOREGROUND}, {@link android.net.NetworkCapabilities#NET_CAPABILITY_NOT_CONGESTED}, {@link android.net.NetworkCapabilities#NET_CAPABILITY_NOT_SUSPENDED}, android.net.NetworkCapabilities.NET_CAPABILITY_OEM_PAID, {@link android.net.NetworkCapabilities#NET_CAPABILITY_MCX}, android.net.NetworkCapabilities.NET_CAPABILITY_PARTIAL_CONNECTIVITY, {@link android.net.NetworkCapabilities#NET_CAPABILITY_TEMPORARILY_NOT_METERED}, android.net.NetworkCapabilities.NET_CAPABILITY_OEM_PRIVATE, android.net.NetworkCapabilities.NET_CAPABILITY_VEHICLE_INTERNAL, android.net.NetworkCapabilities.NET_CAPABILITY_NOT_VCN_MANAGED, {@link android.net.NetworkCapabilities#NET_CAPABILITY_ENTERPRISE}, android.net.NetworkCapabilities.NET_CAPABILITY_VSIM, android.net.NetworkCapabilities.NET_CAPABILITY_BIP, {@link android.net.NetworkCapabilities#NET_CAPABILITY_HEAD_UNIT}, {@link android.net.NetworkCapabilities#NET_CAPABILITY_MMTEL}, {@link android.net.NetworkCapabilities#NET_CAPABILITY_PRIORITIZE_LATENCY}, {@link android.net.NetworkCapabilities#NET_CAPABILITY_PRIORITIZE_BANDWIDTH}, {@link android.net.NetworkCapabilities#NET_CAPABILITY_LOCAL_NETWORK}, or android.net.NetworkCapabilities.NET_CAPABILITY_NOT_BANDWIDTH_CONSTRAINED
 * @return The builder to facilitate chaining.
 */

public android.net.NetworkRequest.Builder removeCapability(int capability) { throw new RuntimeException("Stub!"); }

/**
 * Completely clears all the {@code NetworkCapabilities} from this builder instance,
 * removing even the capabilities that are set by default when the object is constructed.
 * Also removes any set forbidden capabilities.
 *
 * @return The builder to facilitate chaining.

 * This value cannot be {@code null}.
 */

@android.annotation.NonNull
public android.net.NetworkRequest.Builder clearCapabilities() { throw new RuntimeException("Stub!"); }

/**
 * Adds the given transport requirement to this builder.  These represent
 * the set of allowed transports for the request.  Only networks using one
 * of these transports will satisfy the request.  If no particular transports
 * are required, none should be specified here.
 *
 * @param transportType The transport type to add.
 * Value is {@link android.net.NetworkCapabilities#TRANSPORT_CELLULAR}, {@link android.net.NetworkCapabilities#TRANSPORT_WIFI}, {@link android.net.NetworkCapabilities#TRANSPORT_BLUETOOTH}, {@link android.net.NetworkCapabilities#TRANSPORT_ETHERNET}, {@link android.net.NetworkCapabilities#TRANSPORT_VPN}, {@link android.net.NetworkCapabilities#TRANSPORT_WIFI_AWARE}, {@link android.net.NetworkCapabilities#TRANSPORT_LOWPAN}, android.net.NetworkCapabilities.TRANSPORT_TEST, {@link android.net.NetworkCapabilities#TRANSPORT_USB}, {@link android.net.NetworkCapabilities#TRANSPORT_THREAD}, or {@link android.net.NetworkCapabilities#TRANSPORT_SATELLITE}
 * @return The builder to facilitate chaining.
 */

public android.net.NetworkRequest.Builder addTransportType(int transportType) { throw new RuntimeException("Stub!"); }

/**
 * Removes (if found) the given transport from this builder instance.
 *
 * @param transportType The transport type to remove.
 * Value is {@link android.net.NetworkCapabilities#TRANSPORT_CELLULAR}, {@link android.net.NetworkCapabilities#TRANSPORT_WIFI}, {@link android.net.NetworkCapabilities#TRANSPORT_BLUETOOTH}, {@link android.net.NetworkCapabilities#TRANSPORT_ETHERNET}, {@link android.net.NetworkCapabilities#TRANSPORT_VPN}, {@link android.net.NetworkCapabilities#TRANSPORT_WIFI_AWARE}, {@link android.net.NetworkCapabilities#TRANSPORT_LOWPAN}, android.net.NetworkCapabilities.TRANSPORT_TEST, {@link android.net.NetworkCapabilities#TRANSPORT_USB}, {@link android.net.NetworkCapabilities#TRANSPORT_THREAD}, or {@link android.net.NetworkCapabilities#TRANSPORT_SATELLITE}
 * @return The builder to facilitate chaining.
 */

public android.net.NetworkRequest.Builder removeTransportType(int transportType) { throw new RuntimeException("Stub!"); }

/**
 * Sets the optional bearer specific network specifier.
 * This has no meaning if a single transport is also not specified, so calling
 * this without a single transport set will generate an exception, as will
 * subsequently adding or removing transports after this is set.
 * </p>
 * If the {@code networkSpecifier} is provided, it shall be interpreted as follows:
 * <ul>
 * <li>If the specifier can be parsed as an integer, it will be treated as a
 * {@link android.net TelephonyNetworkSpecifier}, and the provided integer will be
 * interpreted as a SubscriptionId.
 * <li>If the value is an ethernet interface name, it will be treated as such.
 * <li>For all other cases, the behavior is undefined.
 * </ul>
 *
 * @param networkSpecifier A {@code String} of either a SubscriptionId in cellular
 *                         network request or an ethernet interface name in ethernet
 *                         network request.
 *
 * @deprecated Use {@link #setNetworkSpecifier(android.net.NetworkSpecifier)} instead.
 */

@Deprecated
public android.net.NetworkRequest.Builder setNetworkSpecifier(java.lang.String networkSpecifier) { throw new RuntimeException("Stub!"); }

/**
 * Sets the optional bearer specific network specifier.
 * This has no meaning if a single transport is also not specified, so calling
 * this without a single transport set will generate an exception, as will
 * subsequently adding or removing transports after this is set.
 * </p>
 *
 * @param networkSpecifier A concrete, parcelable framework class that extends
 *                         NetworkSpecifier.
 */

public android.net.NetworkRequest.Builder setNetworkSpecifier(android.net.NetworkSpecifier networkSpecifier) { throw new RuntimeException("Stub!"); }

/**
 * Sets the optional subscription ID set.
 * <p>
 * This specify the subscription IDs requirement.
 * A network will satisfy this request only if it matches one of the subIds in this set.
 * An empty set matches all networks, including those without a subId.
 *
 * <p>Registering a NetworkRequest with a non-empty set of subIds requires the
 * NETWORK_FACTORY permission.
 *
 * @param subIds A {@code Set} that represents subscription IDs.
 
 * This value cannot be {@code null}.
 * @return This value cannot be {@code null}.
 */

@android.annotation.NonNull
public android.net.NetworkRequest.Builder setSubscriptionIds(@android.annotation.NonNull java.util.Set<java.lang.Integer> subIds) { throw new RuntimeException("Stub!"); }

/**
 * Specifies whether the built request should also match networks that do not apply to the
 * calling UID.
 *
 * By default, the built request will only match networks that apply to the calling UID.
 * If this method is called with {@code true}, the built request will match any network on
 * the system that matches the other parameters of the request. In this case, any
 * information in the built request that is subject to redaction for security or privacy
 * purposes, such as a {@link android.net.NetworkSpecifier NetworkSpecifier}, will be redacted or removed to prevent the
 * application deducing sensitive information.
 *
 * @param include Whether to match networks that do not apply to the calling UID.
 * @return The builder to facilitate chaining.

 * This value cannot be {@code null}.
 */

@android.annotation.NonNull
public android.net.NetworkRequest.Builder setIncludeOtherUidNetworks(boolean include) { throw new RuntimeException("Stub!"); }
}

}

