/*
 * Copyright (C) 2020 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package android.net.vcn;

import android.content.Context;
import android.content.pm.PackageManager;
import java.io.IOException;
import java.util.concurrent.Executor;

/**
 * VcnManager publishes APIs for applications to configure and manage Virtual Carrier Networks.
 *
 * <p>A VCN creates a virtualization layer to allow carriers to aggregate heterogeneous physical
 * networks, unifying them as a single carrier network. This enables infrastructure flexibility on
 * the part of carriers without impacting user connectivity, abstracting the physical network
 * technologies as an implementation detail of their public network.
 *
 * <p>Each VCN virtualizes a carrier's network by building tunnels to a carrier's core network over
 * carrier-managed physical links and supports a IP mobility layer to ensure seamless transitions
 * between the underlying networks. Each VCN is configured based on a Subscription Group (see {@link
 * android.telephony.SubscriptionManager}) and aggregates all networks that are brought up based on
 * a profile or suggestion in the specified Subscription Group.
 *
 * <p>The VCN can be configured to expose one or more {@link android.net.Network}(s), each with
 * different capabilities, allowing for APN virtualization.
 *
 * <p>If a tunnel fails to connect, or otherwise encounters a fatal error, the VCN will attempt to
 * reestablish the connection. If the tunnel still has not reconnected after a system-determined
 * timeout, the VCN Safe Mode (see below) will be entered.
 *
 * <p>The VCN Safe Mode ensures that users (and carriers) have a fallback to restore system
 * connectivity to update profiles, diagnose issues, contact support, or perform other remediation
 * tasks. In Safe Mode, the system will allow underlying cellular networks to be used as default.
 * Additionally, during Safe Mode, the VCN will continue to retry the connections, and will
 * automatically exit Safe Mode if all active tunnels connect successfully.
 *
 * <p>Apps targeting Android 15 or newer should check the existence of {@link android.content.pm.PackageManager#FEATURE_TELEPHONY_SUBSCRIPTION  } before querying the service. If the feature is
 * absent, {@link android.content.Context#getSystemService Context.getSystemService} may return null.

 * <br>
 * Requires the {@link android.content.pm.PackageManager#FEATURE_TELEPHONY_SUBSCRIPTION PackageManager#FEATURE_TELEPHONY_SUBSCRIPTION} feature which can be detected using {@link android.content.pm.PackageManager#hasSystemFeature(String) PackageManager.hasSystemFeature(String)}.
 */

@SuppressWarnings({"unchecked", "deprecation", "all"})
public class VcnManager {

VcnManager() { throw new RuntimeException("Stub!"); }

/**
 * Clears the VCN configuration for a given subscription group.
 *
 * <p>An app that has carrier privileges for any of the subscriptions in the given group may
 * clear a VCN configuration. This API is ONLY permitted for callers running as the primary
 * user. Any active VCN associated with this configuration will be torn down.
 *
 * <br>
 * Requires {@link android.telephony.TelephonyManager#hasCarrierPrivileges carrier privileges}
 * @param subscriptionGroup the subscription group that the configuration should be applied to
 * This value cannot be {@code null}.
 * @throws java.lang.SecurityException if the caller does not have carrier privileges, is not the owner of
 *     the associated configuration, or is not running as the primary user
 * @throws java.io.IOException if the configuration failed to be cleared from disk. This may occur due
 *     to temporary disk errors, or more permanent conditions such as a full disk.
 */

public void clearVcnConfig(@android.annotation.NonNull android.os.ParcelUuid subscriptionGroup) throws java.io.IOException { throw new RuntimeException("Stub!"); }

/**
 * Retrieves the list of Subscription Groups for which a VCN Configuration has been set.
 *
 * <p>The returned list will include only subscription groups for which an associated {@link android.net.vcn.VcnConfig VcnConfig} exists, and the app is either:
 *
 * <ul>
 *   <li>Carrier privileged for that subscription group, or
 *   <li>Is the provisioning package of the config
 * </ul>
 *
 * @throws java.lang.SecurityException if the caller is not running as the primary user

 * @return This value cannot be {@code null}.
 */

@android.annotation.NonNull
public java.util.List<android.os.ParcelUuid> getConfiguredSubscriptionGroups() { throw new RuntimeException("Stub!"); }

/**
 * Registers the given callback to receive status updates for the specified subscription.
 *
 * <p>Callbacks can be registered for a subscription before {@link android.net.vcn.VcnConfig VcnConfig}s are set for it.
 *
 * <p>A {@link android.net.vcn.VcnManager.VcnStatusCallback VcnStatusCallback} may only be registered for one subscription at a time. {@link android.net.vcn.VcnManager.VcnStatusCallback VcnStatusCallback}s may be reused once unregistered.
 *
 * <p>A {@link android.net.vcn.VcnManager.VcnStatusCallback VcnStatusCallback} will only be invoked if the registering package has carrier
 * privileges for the specified subscription at the time of invocation.
 *
 * <p>A {@link android.net.vcn.VcnManager.VcnStatusCallback VcnStatusCallback} is eligible to begin receiving callbacks once it is registered
 * and there is a VCN active for its specified subscription group (this may happen after the
 * callback is registered).
 *
 * <p>{@link android.net.vcn.VcnManager.VcnStatusCallback#onStatusChanged(int) VcnStatusCallback.onStatusChanged(int)} will be invoked on registration with the
 * current status for the specified subscription group's VCN. If the registrant is not
 * privileged for this subscription group, {@link #VCN_STATUS_CODE_NOT_CONFIGURED} will be
 * returned.
 *
 * @param subscriptionGroup The subscription group to match for callbacks
 * This value cannot be {@code null}.
 * @param executor The {@link java.util.concurrent.Executor Executor} to be used for invoking callbacks
 * This value cannot be {@code null}.
 * @param callback The VcnStatusCallback to be registered
 * This value cannot be {@code null}.
 * @throws java.lang.IllegalStateException if callback is currently registered with VcnManager
 */

public void registerVcnStatusCallback(@android.annotation.NonNull android.os.ParcelUuid subscriptionGroup, @android.annotation.NonNull java.util.concurrent.Executor executor, @android.annotation.NonNull android.net.vcn.VcnManager.VcnStatusCallback callback) { throw new RuntimeException("Stub!"); }

/**
 * Sets the VCN configuration for a given subscription group.
 *
 * <p>An app that has carrier privileges for any of the subscriptions in the given group may set
 * a VCN configuration. If a configuration already exists for the given subscription group, it
 * will be overridden. Any active VCN(s) may be forced to restart to use the new configuration.
 *
 * <p>This API is ONLY permitted for callers running as the primary user.
 *
 * <br>
 * Requires {@link android.telephony.TelephonyManager#hasCarrierPrivileges carrier privileges}
 * @param subscriptionGroup the subscription group that the configuration should be applied to
 * This value cannot be {@code null}.
 * @param config the configuration parameters for the VCN
 * This value cannot be {@code null}.
 * @throws java.lang.SecurityException if the caller does not have carrier privileges for the provided
 *     subscriptionGroup, or is not running as the primary user
 * @throws java.io.IOException if the configuration failed to be saved and persisted to disk. This may
 *     occur due to temporary disk errors, or more permanent conditions such as a full disk.
 */

public void setVcnConfig(@android.annotation.NonNull android.os.ParcelUuid subscriptionGroup, @android.annotation.NonNull android.net.vcn.VcnConfig config) throws java.io.IOException { throw new RuntimeException("Stub!"); }

/**
 * Unregisters the given callback.
 *
 * <p>Once unregistered, the callback will stop receiving status updates for the subscription it
 * was registered with.
 *
 * @param callback The callback to be unregistered

 * This value cannot be {@code null}.
 */

public void unregisterVcnStatusCallback(@android.annotation.NonNull android.net.vcn.VcnManager.VcnStatusCallback callback) { throw new RuntimeException("Stub!"); }

/**
 * Value indicating that an error with this Gateway Connection's configuration occurred.
 *
 * <p>For example, this error code will be returned after authentication failures.
 */

public static final int VCN_ERROR_CODE_CONFIG_ERROR = 1; // 0x1

/**
 * Value indicating that an internal failure occurred in this Gateway Connection.
 */

public static final int VCN_ERROR_CODE_INTERNAL_ERROR = 0; // 0x0

/**
 * Value indicating that a Network error occurred with this Gateway Connection.
 *
 * <p>For example, this error code will be returned if an underlying {@link android.net.Network}
 * for this Gateway Connection is lost, or if an error occurs while resolving the connection
 * endpoint address.
 */

public static final int VCN_ERROR_CODE_NETWORK_ERROR = 2; // 0x2

/**
 * Value indicating that the VCN for the subscription group is active.
 *
 * <p>A VCN is active if a {@link android.net.vcn.VcnConfig VcnConfig} is present for the subscription, the provisioning
 * package is privileged, and the VCN is not in Safe Mode. In other words, a VCN is considered
 * active while it is connecting, fully connected, and disconnecting.
 */

public static final int VCN_STATUS_CODE_ACTIVE = 2; // 0x2

/**
 * Value indicating that the VCN for the subscription group is inactive.
 *
 * <p>A VCN is inactive if a {@link android.net.vcn.VcnConfig VcnConfig} is present for the subscription group, but the
 * provisioning package is not privileged.
 */

public static final int VCN_STATUS_CODE_INACTIVE = 1; // 0x1

/**
 * Value indicating that the VCN for the subscription group is not configured, or that the
 * callback is not privileged for the subscription group.
 */

public static final int VCN_STATUS_CODE_NOT_CONFIGURED = 0; // 0x0

/**
 * Value indicating that the VCN for the subscription group is in Safe Mode.
 *
 * <p>A VCN will be put into Safe Mode if any of the gateway connections were unable to
 * establish a connection within a system-determined timeout (while underlying networks were
 * available).
 */

public static final int VCN_STATUS_CODE_SAFE_MODE = 3; // 0x3
/**
 * VcnStatusCallback is the interface for Carrier apps to receive updates for their VCNs.
 *
 * <p>VcnStatusCallbacks may be registered before {@link android.net.vcn.VcnConfig VcnConfig}s are provided for a
 * subscription group.
 */

@SuppressWarnings({"unchecked", "deprecation", "all"})
public abstract static class VcnStatusCallback {

public VcnStatusCallback() { throw new RuntimeException("Stub!"); }

/**
 * Invoked when a VCN Gateway Connection corresponding to this callback's subscription group
 * encounters an error.
 *
 * @param gatewayConnectionName the String GatewayConnection name for the GatewayConnection
 *     encountering an error. This will match the name for exactly one {@link android.net.vcn.VcnGatewayConnectionConfig VcnGatewayConnectionConfig} for the {@link android.net.vcn.VcnConfig VcnConfig} configured for this callback's
 *     subscription group
 * This value cannot be {@code null}.
 * @param errorCode the code to indicate the error that occurred. This value will be one of
 *     VCN_ERROR_CODE_*.
 * Value is {@link android.net.vcn.VcnManager#VCN_ERROR_CODE_INTERNAL_ERROR}, {@link android.net.vcn.VcnManager#VCN_ERROR_CODE_CONFIG_ERROR}, or {@link android.net.vcn.VcnManager#VCN_ERROR_CODE_NETWORK_ERROR}
 * @param detail Throwable to provide additional information about the error, or {@code
 *     null} if none
 */

public abstract void onGatewayConnectionError(@android.annotation.NonNull java.lang.String gatewayConnectionName, int errorCode, @android.annotation.Nullable java.lang.Throwable detail);

/**
 * Invoked when status of the VCN for this callback's subscription group changes.
 *
 * @param statusCode the code for the status change encountered by this {@link android.net.vcn.VcnManager.VcnStatusCallback VcnStatusCallback}'s subscription group. This value will be one of VCN_STATUS_CODE_*.

 * Value is {@link android.net.vcn.VcnManager#VCN_STATUS_CODE_NOT_CONFIGURED}, {@link android.net.vcn.VcnManager#VCN_STATUS_CODE_INACTIVE}, {@link android.net.vcn.VcnManager#VCN_STATUS_CODE_ACTIVE}, or {@link android.net.vcn.VcnManager#VCN_STATUS_CODE_SAFE_MODE}
 */

public abstract void onStatusChanged(int statusCode);
}

}

