/*
 * Copyright (C) 2019 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package android.net.ipsec.ike;

import android.content.Context;
import java.util.concurrent.Executor;

/**
 * This class represents an IKE Session management object that allows for keying and management of
 * {@link android.net.IpSecTransform}s.
 *
 * <p>An IKE/Child Session represents an IKE/Child SA as well as its rekeyed successors. A Child
 * Session is bounded by the lifecycle of the IKE Session under which it is set up. Closing an IKE
 * Session implicitly closes any remaining Child Sessions under it.
 *
 * <p>An IKE procedure is one or multiple IKE message exchanges that are used to create, delete or
 * rekey an IKE Session or Child Session.
 *
 * <p>This class provides methods for initiating IKE procedures, such as the Creation and Deletion
 * of a Child Session, or the Deletion of the IKE session. All procedures (except for IKE deletion)
 * will be initiated sequentially after IKE Session is set up.
 *
 * @see <a href="https://tools.ietf.org/html/rfc7296">RFC 7296, Internet Key Exchange Protocol
 *     Version 2 (IKEv2)</a>
 */

@SuppressWarnings({"unchecked", "deprecation", "all"})
public final class IkeSession implements java.lang.AutoCloseable {

/**
 * Constructs a new IKE session.
 *
 * <p>This method will immediately return an instance of {@link android.net.ipsec.ike.IkeSession IkeSession} and asynchronously
 * initiate the setup procedure of {@link android.net.ipsec.ike.IkeSession IkeSession} as well as its first Child Session.
 * Callers will be notified of these two setup results via the callback arguments.
 *
 * <p>FEATURE_IPSEC_TUNNELS is required for setting up a tunnel mode Child SA.
 *
 * @param context a valid {@link android.content.Context Context} instance.
 * @param ikeSessionParams the {@link android.net.ipsec.ike.IkeSessionParams IkeSessionParams} that contains a set of valid {@link android.net.ipsec.ike.IkeSession IkeSession} configurations.
 * @param firstChildSessionParams the {@link android.net.ipsec.ike.ChildSessionParams ChildSessionParams} that contains a set of valid
 *     configurations for the first Child Session.
 * @param userCbExecutor the {@link java.util.concurrent.Executor Executor} upon which all callbacks will be posted. For
 *     security and consistency, the callbacks posted to this executor MUST be executed serially
 *     and in the order they were posted, as guaranteed by executors such as {@link
 *     java.util.concurrent.Executors#newSingleThreadExecutor()}
 * @param ikeSessionCallback the {@link android.net.ipsec.ike.IkeSessionCallback IkeSessionCallback} interface to notify callers of state
 *     changes within the {@link android.net.ipsec.ike.IkeSession IkeSession}.
 * @param firstChildSessionCallback the {@link android.net.ipsec.ike.ChildSessionCallback ChildSessionCallback} interface to notify callers
 *     of state changes within the first Child Session.
 * @return an instance of {@link android.net.ipsec.ike.IkeSession IkeSession}.
 */

public IkeSession(@android.annotation.NonNull android.content.Context context, @android.annotation.NonNull android.net.ipsec.ike.IkeSessionParams ikeSessionParams, @android.annotation.NonNull android.net.ipsec.ike.ChildSessionParams firstChildSessionParams, @android.annotation.NonNull java.util.concurrent.Executor userCbExecutor, @android.annotation.NonNull android.net.ipsec.ike.IkeSessionCallback ikeSessionCallback, @android.annotation.NonNull android.net.ipsec.ike.ChildSessionCallback firstChildSessionCallback) { throw new RuntimeException("Stub!"); }

/** @hide */

public void finalize() { throw new RuntimeException("Stub!"); }

/**
 * Request a new Child Session.
 *
 * <p>Users MUST provide a unique {@link android.net.ipsec.ike.ChildSessionCallback ChildSessionCallback} instance for each new Child
 * Session.
 *
 * <p>Upon setup, {@link android.net.ipsec.ike.ChildSessionCallback#onOpened(android.net.ipsec.ike.ChildSessionConfiguration) ChildSessionCallback#onOpened(ChildSessionConfiguration)} will be
 * fired.
 *
 * <p>FEATURE_IPSEC_TUNNELS is required for setting up a tunnel mode Child SA.
 *
 * @param childSessionParams the {@link android.net.ipsec.ike.ChildSessionParams ChildSessionParams} that contains the Child Session
 *     configurations to negotiate.
 * @param childSessionCallback the {@link android.net.ipsec.ike.ChildSessionCallback ChildSessionCallback} interface to notify users the
 *     state changes of the Child Session. It will be posted to the callback {@link java.util.concurrent.Executor Executor} of
 *     this {@link android.net.ipsec.ike.IkeSession IkeSession}.
 * @throws java.lang.IllegalArgumentException if the ChildSessionCallback is already in use.
 */

public void openChildSession(@android.annotation.NonNull android.net.ipsec.ike.ChildSessionParams childSessionParams, @android.annotation.NonNull android.net.ipsec.ike.ChildSessionCallback childSessionCallback) { throw new RuntimeException("Stub!"); }

/**
 * Delete a Child Session.
 *
 * <p>Upon closure, {@link android.net.ipsec.ike.ChildSessionCallback#onClosed() ChildSessionCallback#onClosed()} will be fired.
 *
 * @param childSessionCallback The {@link android.net.ipsec.ike.ChildSessionCallback ChildSessionCallback} instance that uniquely identify
 *     the Child Session.
 * @throws java.lang.IllegalArgumentException if no Child Session found bound with this callback.
 */

public void closeChildSession(@android.annotation.NonNull android.net.ipsec.ike.ChildSessionCallback childSessionCallback) { throw new RuntimeException("Stub!"); }

/**
 * Close the IKE session gracefully.
 *
 * <p>Implements {@link java.lang.AutoCloseable#close() AutoCloseable#close()}
 *
 * <p>Upon closure, {@link android.net.ipsec.ike.IkeSessionCallback#onClosed() IkeSessionCallback#onClosed()} or {@link android.net.ipsec.ike.IkeSessionCallback#onClosedWithException(IkeException)  } will be fired.
 *
 * <p>Closing an IKE Session implicitly closes any remaining Child Sessions negotiated under it.
 * Users SHOULD stop all outbound traffic that uses these Child Sessions ({@link
 * android.net.IpSecTransform} pairs) before calling this method. Otherwise IPsec packets will
 * be dropped due to the lack of a valid {@link android.net.IpSecTransform}.
 *
 * <p>Closure of an IKE session will take priority over, and cancel other procedures waiting in
 * the queue (but will wait for ongoing locally initiated procedures to complete). After sending
 * the Delete request, the IKE library will wait until a Delete response is received or
 * retransmission timeout occurs.
 */

public void close() { throw new RuntimeException("Stub!"); }

/**
 * Terminate (forcibly close) the IKE session.
 *
 * <p>Upon closing, {@link android.net.ipsec.ike.IkeSessionCallback#onClosed() IkeSessionCallback#onClosed()} will be fired.
 *
 * <p>Closing an IKE Session implicitly closes any remaining Child Sessions negotiated under it.
 * Users SHOULD stop all outbound traffic that uses these Child Sessions ({@link
 * android.net.IpSecTransform} pairs) before calling this method. Otherwise IPsec packets will
 * be dropped due to the lack of a valid {@link android.net.IpSecTransform}.
 *
 * <p>Forcible closure of an IKE session will take priority over, and cancel other procedures
 * waiting in the queue. It will also interrupt any ongoing locally initiated procedure.
 */

public void kill() { throw new RuntimeException("Stub!"); }
}

