/*
 * Copyright (C) 2020 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */


package android.media;

import android.os.Bundle;
import java.util.List;
import org.xmlpull.v1.XmlPullParser;

/**
 ApplicationMediaCapabilities is an immutable class that encapsulates an application's capabilities
 for handling newer video codec format and media features.
 
 <p>
 Android 12 introduces Compatible media transcoding feature.  See
 <a href="https://developer.android.com/about/versions/12/features#compatible_media_transcoding">
 Compatible media transcoding</a>. By default, Android assumes apps can support playback of all
 media formats. Apps that would like to request that media be transcoded into a more compatible
 format should declare their media capabilities in a media_capabilities.xml resource file and add it
 as a property tag in the AndroidManifest.xml file. Here is a example:
 <pre>
 {@code
 <media-capabilities xmlns:android="http://schemas.android.com/apk/res/android">
    <format android:name="HEVC" supported="true"/>
    <format android:name="HDR10" supported="false"/>
    <format android:name="HDR10Plus" supported="false"/>
 </media-capabilities>
 }
 </pre>
 The ApplicationMediaCapabilities class is generated from this xml and used by the platform to
 represent an application's media capabilities in order to determine whether modern media files need
 to be transcoded for that application.
 </p>
 
 <p>
 ApplicationMediaCapabilities objects can also be built by applications at runtime for use with
 {@link android.content.ContentResolver#openTypedAssetFileDescriptor(android.net.Uri,java.lang.String,android.os.Bundle) ContentResolver#openTypedAssetFileDescriptor(Uri, String, Bundle)} to provide more
 control over the transcoding that is built into the platform. ApplicationMediaCapabilities
 provided by applications at runtime like this override the default manifest capabilities for that
 media access.The object could be build either through {@link #createFromXml(org.xmlpull.v1.XmlPullParser)} or
 through the builder class {@link android.media.ApplicationMediaCapabilities.Builder ApplicationMediaCapabilities.Builder}
 
 <h3> Video Codec Support</h3>
 <p>
 Newer video codes include HEVC, VP9 and AV1. Application only needs to indicate their support
 for newer format with this class as they are assumed to support older format like h.264.
 
 <h3>Capability of handling HDR(high dynamic range) video</h3>
 <p>
 There are four types of HDR video(Dolby-Vision, HDR10, HDR10+, HLG) supported by the platform,
 application will only need to specify individual types they supported.
 */

@SuppressWarnings({"unchecked", "deprecation", "all"})
public final class ApplicationMediaCapabilities implements android.os.Parcelable {

ApplicationMediaCapabilities() { throw new RuntimeException("Stub!"); }

/**
 * Query if a video codec format is supported by the application.
 * <p>
 * If the application has not specified supporting the format or not, this will return false.
 * Use {@link #isFormatSpecified(java.lang.String)} to query if a format is specified or not.
 *
 * @param videoMime The mime type of the video codec format. Must be the one used in
 * {@link android.media.MediaFormat#KEY_MIME MediaFormat#KEY_MIME}.
 * This value cannot be {@code null}.
 * @return true if application supports the video codec format, false otherwise.
 */

public boolean isVideoMimeTypeSupported(@android.annotation.NonNull java.lang.String videoMime) { throw new RuntimeException("Stub!"); }

/**
 * Query if a HDR type is supported by the application.
 * <p>
 * If the application has not specified supporting the format or not, this will return false.
 * Use {@link #isFormatSpecified(java.lang.String)} to query if a format is specified or not.
 *
 * @param hdrType The type of the HDR format.
 * This value cannot be {@code null}.
 * Value is {@link android.media.MediaFeature.HdrType#DOLBY_VISION}, {@link android.media.MediaFeature.HdrType#HDR10}, {@link android.media.MediaFeature.HdrType#HDR10_PLUS}, or {@link android.media.MediaFeature.HdrType#HLG}
 * @return true if application supports the HDR format, false otherwise.
 */

public boolean isHdrTypeSupported(@android.annotation.NonNull java.lang.String hdrType) { throw new RuntimeException("Stub!"); }

/**
 * Query if a format is specified by the application.
 * <p>
 * The format could be either the video format or the hdr format.
 *
 * @param format The name of the format.
 * This value cannot be {@code null}.
 * @return true if application specifies the format, false otherwise.
 */

public boolean isFormatSpecified(@android.annotation.NonNull java.lang.String format) { throw new RuntimeException("Stub!"); }

public int describeContents() { throw new RuntimeException("Stub!"); }

/**
 * {@inheritDoc}

 * @param dest This value cannot be {@code null}.
 */

public void writeToParcel(@android.annotation.NonNull android.os.Parcel dest, int flags) { throw new RuntimeException("Stub!"); }

public java.lang.String toString() { throw new RuntimeException("Stub!"); }

/**
 * Query the video codec mime types supported by the application.
 * @return List of supported video codec mime types. The list will be empty if there are none.

 * This value cannot be {@code null}.
 */

@android.annotation.NonNull
public java.util.List<java.lang.String> getSupportedVideoMimeTypes() { throw new RuntimeException("Stub!"); }

/**
 * Query the video codec mime types that are not supported by the application.
 * @return List of unsupported video codec mime types. The list will be empty if there are none.

 * This value cannot be {@code null}.
 */

@android.annotation.NonNull
public java.util.List<java.lang.String> getUnsupportedVideoMimeTypes() { throw new RuntimeException("Stub!"); }

/**
 * Query all hdr types that are supported by the application.
 * @return List of supported hdr types. The list will be empty if there are none.

 * This value cannot be {@code null}.
 */

@android.annotation.NonNull
public java.util.List<java.lang.String> getSupportedHdrTypes() { throw new RuntimeException("Stub!"); }

/**
 * Query all hdr types that are not supported by the application.
 * @return List of unsupported hdr types. The list will be empty if there are none.

 * This value cannot be {@code null}.
 */

@android.annotation.NonNull
public java.util.List<java.lang.String> getUnsupportedHdrTypes() { throw new RuntimeException("Stub!"); }

/**
 * Creates {@link android.media.ApplicationMediaCapabilities ApplicationMediaCapabilities} from an xml.
 *
 * The xml's syntax is the same as the media_capabilities.xml used by the AndroidManifest.xml.
 * <p> Here is an example:
 *
 * <pre>
 * {@code
 * <media-capabilities xmlns:android="http://schemas.android.com/apk/res/android">
 *     <format android:name="HEVC" supported="true"/>
 *     <format android:name="HDR10" supported="false"/>
 *     <format android:name="HDR10Plus" supported="false"/>
 * </media-capabilities>
 * }
 * </pre>
 * <p>
 *
 * @param xmlParser The underlying {@link org.xmlpull.v1.XmlPullParser XmlPullParser} that will read the xml.
 * This value cannot be {@code null}.
 * @return An ApplicationMediaCapabilities object.
 * This value cannot be {@code null}.
 * @throws java.lang.UnsupportedOperationException if the capabilities in xml config are invalid or
 * incompatible.
 */

@android.annotation.NonNull
public static android.media.ApplicationMediaCapabilities createFromXml(@android.annotation.NonNull org.xmlpull.v1.XmlPullParser xmlParser) { throw new RuntimeException("Stub!"); }

@android.annotation.NonNull public static final android.os.Parcelable.Creator<android.media.ApplicationMediaCapabilities> CREATOR;
static { CREATOR = null; }
/**
 * Builder class for {@link android.media.ApplicationMediaCapabilities ApplicationMediaCapabilities} objects.
 * Use this class to configure and create an ApplicationMediaCapabilities instance. Builder
 * could be created from an existing ApplicationMediaCapabilities object, from a xml file or
 * MediaCodecList.
 * //TODO(hkuang): Add xml parsing support to the builder.
 */

@SuppressWarnings({"unchecked", "deprecation", "all"})
public static final class Builder {

/**
 * Constructs a new Builder with all the supports default to false.
 */

public Builder() { throw new RuntimeException("Stub!"); }

/**
 * Builds a {@link android.media.ApplicationMediaCapabilities ApplicationMediaCapabilities} object.
 *
 * @return a new {@link android.media.ApplicationMediaCapabilities ApplicationMediaCapabilities} instance successfully initialized
 * with all the parameters set on this <code>Builder</code>.
 * This value cannot be {@code null}.
 * @throws java.lang.UnsupportedOperationException if the parameters set on the
 *                                       <code>Builder</code> were incompatible, or if they
 *                                       are not supported by the
 *                                       device.
 */

@android.annotation.NonNull
public android.media.ApplicationMediaCapabilities build() { throw new RuntimeException("Stub!"); }

/**
 * Adds a supported video codec mime type.
 *
 * @param codecMime Supported codec mime types. Must be one of the mime type defined
 *                  in {@link android.media.MediaFormat MediaFormat}.
 * This value cannot be {@code null}.
 * @throws java.lang.IllegalArgumentException if mime type is not valid.
 
 * @return This value cannot be {@code null}.
 */

@android.annotation.NonNull
public android.media.ApplicationMediaCapabilities.Builder addSupportedVideoMimeType(@android.annotation.NonNull java.lang.String codecMime) { throw new RuntimeException("Stub!"); }

/**
 * Adds an unsupported video codec mime type.
 *
 * @param codecMime Unsupported codec mime type. Must be one of the mime type defined
 *                  in {@link android.media.MediaFormat MediaFormat}.
 * This value cannot be {@code null}.
 * @throws java.lang.IllegalArgumentException if mime type is not valid.
 
 * @return This value cannot be {@code null}.
 */

@android.annotation.NonNull
public android.media.ApplicationMediaCapabilities.Builder addUnsupportedVideoMimeType(@android.annotation.NonNull java.lang.String codecMime) { throw new RuntimeException("Stub!"); }

/**
 * Adds a supported hdr type.
 *
 * @param hdrType Supported hdr type. Must be one of the String defined in
 *                {@link android.media.MediaFeature.HdrType MediaFeature.HdrType}.
 * This value cannot be {@code null}.
 * Value is {@link android.media.MediaFeature.HdrType#DOLBY_VISION}, {@link android.media.MediaFeature.HdrType#HDR10}, {@link android.media.MediaFeature.HdrType#HDR10_PLUS}, or {@link android.media.MediaFeature.HdrType#HLG}
 * @throws java.lang.IllegalArgumentException if hdrType is not valid.
 
 * @return This value cannot be {@code null}.
 */

@android.annotation.NonNull
public android.media.ApplicationMediaCapabilities.Builder addSupportedHdrType(@android.annotation.NonNull java.lang.String hdrType) { throw new RuntimeException("Stub!"); }

/**
 * Adds an unsupported hdr type.
 *
 * @param hdrType Unsupported hdr type. Must be one of the String defined in
 *                {@link android.media.MediaFeature.HdrType MediaFeature.HdrType}.
 * This value cannot be {@code null}.
 * Value is {@link android.media.MediaFeature.HdrType#DOLBY_VISION}, {@link android.media.MediaFeature.HdrType#HDR10}, {@link android.media.MediaFeature.HdrType#HDR10_PLUS}, or {@link android.media.MediaFeature.HdrType#HLG}
 * @throws java.lang.IllegalArgumentException if hdrType is not valid.
 
 * @return This value cannot be {@code null}.
 */

@android.annotation.NonNull
public android.media.ApplicationMediaCapabilities.Builder addUnsupportedHdrType(@android.annotation.NonNull java.lang.String hdrType) { throw new RuntimeException("Stub!"); }
}

}

