/*
 * Copyright (C) 2023 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */


package android.graphics.pdf;

import android.graphics.Bitmap;
import android.graphics.Matrix;
import android.graphics.Rect;
import android.graphics.pdf.content.PdfPageGotoLinkContent;
import android.graphics.pdf.content.PdfPageImageContent;
import android.graphics.pdf.content.PdfPageTextContent;
import android.graphics.pdf.models.FormEditRecord;
import android.graphics.pdf.models.FormWidgetInfo;
import android.graphics.pdf.models.PageMatchBounds;
import android.graphics.pdf.models.selection.PageSelection;
import android.graphics.pdf.models.selection.SelectionBoundary;
import android.os.ParcelFileDescriptor;
import java.io.IOException;
import java.util.List;

/**
 * <p>
 * This class enables rendering a PDF document and selecting, searching, fast scrolling,
 * annotations, etc from Android R till Android U. This class is thread safe and can be called by
 * multiple threads however only one thread will be executed at a time as the access is
 * synchronized by internal locking.
 * <p>
 * If you want to render a PDF, you will need to create a new instance of renderer for each
 * document. To render each page, you open the page using the renderer instance created earlier,
 * render it, and close the page. After you are done with rendering, you close the renderer. After
 * the renderer is closed it should not be used anymore. Note that the pages are rendered one by
 * one, i.e. you can have only a single page opened at any given time.
 * <p>
 * A typical use of the APIs to render a PDF looks like this:
 * <pre>
 * // create a new renderer
 * try (PdfRendererPreV renderer = new PdfRendererPreV(getSeekableFileDescriptor(), loadParams)) {
 *      // let us just render all pages
 *      final int pageCount = renderer.getPageCount();
 *      for (int i = 0; i < pageCount; i++) {
 *          Page page = renderer.openPage(i);
 *          RenderParams params = new RenderParams.Builder(Page.RENDER_MODE_FOR_DISPLAY).build();
 *
 *          // say we render for showing on the screen
 *          page.render(mBitmap, params, null, null);
 *
 *          // do stuff with the bitmap
 *
 *          // close the page
 *          page.close();
 *      }
 * }
 * </pre>
 * <h3>Print preview and print output</h3>
 * <p>
 * Please refer to {@link PdfRenderer} for fulfilling this usecase.
 */

@SuppressWarnings({"unchecked", "deprecation", "all"})
public final class PdfRendererPreV implements java.lang.AutoCloseable {

/**
 * Creates a new instance of PdfRendererPreV class.
 * <p>
 * <strong>Note:</strong> The provided file descriptor must be <strong>seekable</strong>,
 * i.e. its data being randomly accessed, e.g. pointing to a file.
 * <p>
 * <strong>Note:</strong> This class takes ownership of the passed in file descriptor
 * and is responsible for closing it when the renderer is closed.
 * <p>
 * If the file is from an untrusted source it is recommended to run the renderer in a separate,
 * isolated process with minimal permissions to limit the impact of security exploits.
 *
 * @param fileDescriptor Seekable file descriptor to read from.
 * This value cannot be {@code null}.
 * @throws java.io.IOException         If an error occurs while reading the file.
 * @throws java.lang.SecurityException If the file requires a password or
 *                                     the security scheme is not supported by the renderer.
 * @throws java.lang.IllegalArgumentException    If the {@link android.os.ParcelFileDescriptor ParcelFileDescriptor} is not seekable.
 * @throws java.lang.NullPointerException        If the file descriptor is null.
 */

public PdfRendererPreV(@android.annotation.NonNull android.os.ParcelFileDescriptor fileDescriptor) throws java.io.IOException { throw new RuntimeException("Stub!"); }

/**
 * Creates a new instance of PdfRendererPreV class.
 * <p>
 * <strong>Note:</strong> The provided file descriptor must be <strong>seekable</strong>,
 * i.e. its data being randomly accessed, e.g. pointing to a file. If the password passed in
 * {@link android.graphics.pdf.LoadParams} is incorrect, the
 * {@link android.graphics.pdf.PdfRendererPreV} will throw a {@link java.lang.SecurityException SecurityException}.
 * <p>
 * <strong>Note:</strong> This class takes ownership of the passed in file descriptor
 * and is responsible for closing it when the renderer is closed.
 * <p>
 * If the file is from an untrusted source it is recommended to run the renderer in a separate,
 * isolated process with minimal permissions to limit the impact of security exploits.
 *
 * @param fileDescriptor Seekable file descriptor to read from.
 * This value cannot be {@code null}.
 * @param params         Instance of {@link android.graphics.pdf.LoadParams LoadParams} specifying params for loading PDF
 *                       document.
 * This value cannot be {@code null}.
 * @throws java.io.IOException         If an error occurs while reading the file.
 * @throws java.lang.SecurityException If the file requires a password or
 *                                     the security scheme is not supported by the renderer.
 * @throws java.lang.IllegalArgumentException    If the {@link android.os.ParcelFileDescriptor ParcelFileDescriptor} is not seekable.
 * @throws java.lang.NullPointerException        If the file descriptor or load params is null.
 */

public PdfRendererPreV(@android.annotation.NonNull android.os.ParcelFileDescriptor fileDescriptor, @android.annotation.NonNull android.graphics.pdf.LoadParams params) throws java.io.IOException { throw new RuntimeException("Stub!"); }

/**
 * Closes this renderer and destroys any cached instance of the document. You should not use
 * this instance after this method is called.
 *
 * @throws java.lang.IllegalStateException If {@link #close()} is called before invoking this.
 */

public void close() { throw new RuntimeException("Stub!"); }

protected void finalize() throws java.lang.Throwable { throw new RuntimeException("Stub!"); }

/**
 * Gets the type of the PDF document.
 *
 * @return The PDF document type.
 * Value is {@link android.graphics.pdf.PdfRendererPreV#DOCUMENT_LINEARIZED_TYPE_NON_LINEARIZED}, or {@link android.graphics.pdf.PdfRendererPreV#DOCUMENT_LINEARIZED_TYPE_LINEARIZED}
 * @throws java.lang.IllegalStateException If {@link #close()} is called before invoking this.
 */

public int getDocumentLinearizationType() { throw new RuntimeException("Stub!"); }

/**
 * Gets the number of pages in the document.
 *
 * @return The page count.
 * Value is 0 or greater
 * @throws java.lang.IllegalStateException If {@link #close()} is called before invoking this.
 */

public int getPageCount() { throw new RuntimeException("Stub!"); }

/**
 * Returns the form type of the loaded PDF
 *
 * @throws java.lang.IllegalStateException    if the renderer is closed
 * @throws java.lang.IllegalArgumentException if an unexpected PDF form type is returned

 * @return Value is {@link android.graphics.pdf.PdfRendererPreV#PDF_FORM_TYPE_NONE}, {@link android.graphics.pdf.PdfRendererPreV#PDF_FORM_TYPE_ACRO_FORM}, {@link android.graphics.pdf.PdfRendererPreV#PDF_FORM_TYPE_XFA_FULL}, or {@link android.graphics.pdf.PdfRendererPreV#PDF_FORM_TYPE_XFA_FOREGROUND}
 */

public int getPdfFormType() { throw new RuntimeException("Stub!"); }

/**
 * Opens a {@link android.graphics.pdf.PdfRendererPreV.Page Page} for rendering.
 *
 * @param index The page index to open, starting from index 0.
 * Value is 0 or greater
 * @return A page that can be rendered.
 * This value cannot be {@code null}.
 * @throws java.lang.IllegalStateException    If {@link #close()} is called before invoking this.
 * @throws java.lang.IllegalArgumentException If the page number is less than 0 or greater than or equal
 *                                  to the total page count.
 */

@android.annotation.NonNull
public android.graphics.pdf.PdfRendererPreV.Page openPage(int index) { throw new RuntimeException("Stub!"); }

/**
 * <p>
 * Saves the current state of the loaded PDF document to the given writable
 * {@link android.os.ParcelFileDescriptor ParcelFileDescriptor}. If the document is password-protected then setting
 * {@code removePasswordProtection} removes the protection before saving. The PDF document
 * should already be decrypted with the correct password before writing. Useful for printing or
 * sharing.
 * <strong>Note:</strong> This method closes the provided file descriptor.
 *
 * @param destination              The writable {@link android.os.ParcelFileDescriptor ParcelFileDescriptor}
 * This value cannot be {@code null}.
 * @param removePasswordProtection If true, removes password protection from the PDF before
 *                                 saving.
 * @throws java.io.IOException           If there's a write error, or if 'removePasswordSecurity' is
 *                               {@code true} but the document remains encrypted.
 * @throws java.lang.IllegalStateException If {@link #close()} is called before invoking this.
 */

public void write(@android.annotation.NonNull android.os.ParcelFileDescriptor destination, boolean removePasswordProtection) throws java.io.IOException { throw new RuntimeException("Stub!"); }

/** Represents a linearized PDF document. */

public static final int DOCUMENT_LINEARIZED_TYPE_LINEARIZED = 1; // 0x1

/** Represents a non-linearized PDF document. */

public static final int DOCUMENT_LINEARIZED_TYPE_NON_LINEARIZED = 0; // 0x0

/** Represents a PDF with form fields specified using the AcroForm spec */

public static final int PDF_FORM_TYPE_ACRO_FORM = 1; // 0x1

/** Represents a PDF without form fields */

public static final int PDF_FORM_TYPE_NONE = 0; // 0x0

/** Represents a PDF with form fields specified using the XFAF subset of the XFA spec */

public static final int PDF_FORM_TYPE_XFA_FOREGROUND = 3; // 0x3

/** Represents a PDF with form fields specified using the entire XFA spec */

public static final int PDF_FORM_TYPE_XFA_FULL = 2; // 0x2
/**
 * This class represents a PDF document page for rendering.
 */

@SuppressWarnings({"unchecked", "deprecation", "all"})
public final class Page implements java.lang.AutoCloseable {

Page() { throw new RuntimeException("Stub!"); }

/**
 * Applies a {@link android.graphics.pdf.models.FormEditRecord FormEditRecord} to the PDF.
 *
 * <p>Apps must call {@link #render(android.graphics.Bitmap,android.graphics.Rect,android.graphics.Matrix,android.graphics.pdf.RenderParams)} to render new
 * bitmaps for the corresponding areas of the page.
 *
 * <p>For click type {@link android.graphics.pdf.models.FormEditRecord FormEditRecord}s, performs a click on {@link android.graphics.pdf.models.FormEditRecord#getClickPoint()  }
 *
 * <p>For set text type {@link android.graphics.pdf.models.FormEditRecord FormEditRecord}s, sets the text value of the form widget.
 *
 * <p>For set indices type {@link android.graphics.pdf.models.FormEditRecord FormEditRecord}s, sets the {@link android.graphics.pdf.models.FormEditRecord#getSelectedIndices()  } as selected and all others as unselected for the
 * form widget indicated by the record.
 *
 * @param editRecord the {@link android.graphics.pdf.models.FormEditRecord FormEditRecord} to be applied
 * This value cannot be {@code null}.
 * @return Rectangular areas of the page bitmap that have been invalidated by this action.
 * This value cannot be {@code null}.
 * @throws java.lang.IllegalArgumentException if the provided {@link android.graphics.pdf.models.FormEditRecord FormEditRecord} cannot be
 *                                  applied to the widget indicated by the index, or if the
 *                                  index does not correspond to a widget on the page.
 * @throws java.lang.IllegalStateException    If the document is already closed.
 * @throws java.lang.IllegalStateException    If the page is already closed.
 */

@android.annotation.NonNull
public java.util.List<android.graphics.Rect> applyEdit(@android.annotation.NonNull android.graphics.pdf.models.FormEditRecord editRecord) { throw new RuntimeException("Stub!"); }

/**
 * Closes this page.
 *
 * @see android.graphics.pdf.PdfRenderer#openPage(int)
 */

public void close() { throw new RuntimeException("Stub!"); }

protected void finalize() throws java.lang.Throwable { throw new RuntimeException("Stub!"); }

/**
 * Returns information about the widget with {@code annotationIndex}.
 *
 * @param annotationIndex the index of the widget within the page's "Annot" array in the
 *                        PDF document, available on results of previous calls to
 *                        {@link #getFormWidgetInfos(int[])} or
 *                        {@link #getFormWidgetInfoAtPosition(int,int)} via
 *                        {@link android.graphics.pdf.models.FormWidgetInfo#getWidgetIndex() FormWidgetInfo.getWidgetIndex()}.
 * Value is 0 or greater
 * @throws java.lang.IllegalArgumentException if there is no form widget at the provided index.
 * @throws java.lang.IllegalStateException    If the document is already closed.
 * @throws java.lang.IllegalStateException    If the page is already closed.
 
 * @return This value cannot be {@code null}.
 */

@android.annotation.NonNull
public android.graphics.pdf.models.FormWidgetInfo getFormWidgetInfoAtIndex(int annotationIndex) { throw new RuntimeException("Stub!"); }

/**
 * Returns information about the widget at the given point.
 *
 * @param x the x position of the widget on the page, in points
 * @param y the y position of the widget on the page, in points
 * @throws java.lang.IllegalArgumentException if there is no form widget at the provided position.
 * @throws java.lang.IllegalStateException    If the document is already closed.
 * @throws java.lang.IllegalStateException    If the page is already closed.

 * @return This value cannot be {@code null}.
 */

@android.annotation.NonNull
public android.graphics.pdf.models.FormWidgetInfo getFormWidgetInfoAtPosition(int x, int y) { throw new RuntimeException("Stub!"); }

/**
 * Returns information about all form widgets on the page, or an empty list if there are no
 * form widgets on the page.
 *
 * @throws java.lang.IllegalStateException If the document is already closed.
 * @throws java.lang.IllegalStateException If the page is already closed.

 * @return This value cannot be {@code null}.
 */

@android.annotation.NonNull
public java.util.List<android.graphics.pdf.models.FormWidgetInfo> getFormWidgetInfos() { throw new RuntimeException("Stub!"); }

/**
 * Returns information about all form widgets of the specified types on the page, or an
 * empty list if there are no form widgets of the specified types on the page.
 *
 * @param types the types of form widgets to return, or an empty array to return all widgets
 * This value cannot be {@code null}.
 * Value is {@link android.graphics.pdf.models.FormWidgetInfo#WIDGET_TYPE_UNKNOWN}, {@link android.graphics.pdf.models.FormWidgetInfo#WIDGET_TYPE_PUSHBUTTON}, {@link android.graphics.pdf.models.FormWidgetInfo#WIDGET_TYPE_CHECKBOX}, {@link android.graphics.pdf.models.FormWidgetInfo#WIDGET_TYPE_RADIOBUTTON}, {@link android.graphics.pdf.models.FormWidgetInfo#WIDGET_TYPE_COMBOBOX}, {@link android.graphics.pdf.models.FormWidgetInfo#WIDGET_TYPE_LISTBOX}, {@link android.graphics.pdf.models.FormWidgetInfo#WIDGET_TYPE_TEXTFIELD}, or {@link android.graphics.pdf.models.FormWidgetInfo#WIDGET_TYPE_SIGNATURE}
 * @throws java.lang.IllegalStateException If the document is already closed.
 * @throws java.lang.IllegalStateException If the page is already closed.
 
 * @return This value cannot be {@code null}.
 */

@android.annotation.NonNull
public java.util.List<android.graphics.pdf.models.FormWidgetInfo> getFormWidgetInfos(@android.annotation.NonNull int[] types) { throw new RuntimeException("Stub!"); }

/**
 * Gets bookmarks and goto links present on the page of a pdf document. Goto Links
 * are the internal navigation links which directs the user to different location
 * within the same document.
 *
 * @return list of all goto links {@link android.graphics.pdf.content.PdfPageGotoLinkContent PdfPageGotoLinkContent} on a page, ordered
 * left to right and top to bottom.
 * This value cannot be {@code null}.
 * @throws java.lang.IllegalStateException If the document/page is closed before invocation.
 */

@android.annotation.NonNull
public java.util.List<android.graphics.pdf.content.PdfPageGotoLinkContent> getGotoLinks() { throw new RuntimeException("Stub!"); }

/**
 * Returns the height of the given {@link android.graphics.pdf.PdfRendererPreV.Page Page} object in points (1/72"). It is not
 * guaranteed that all pages will have the same height and the viewport should be resized to
 * the given page height.
 *
 * @return height of the given page
 * Value is 0 or greater
 * @throws java.lang.IllegalStateException If the document/page is closed before invocation.
 */

public int getHeight() { throw new RuntimeException("Stub!"); }

/**
 * Return list of {@link android.graphics.pdf.content.PdfPageImageContent PdfPageImageContent} found on the page, ordered left to right
 * and top to bottom. It contains all the content associated with images found on the
 * page including alt text. The list will be empty if there are no results found.
 * Currently, localisation does not have any impact on the order in which
 * {@link android.graphics.pdf.content.PdfPageImageContent PdfPageImageContent} is returned.
 *
 * @return list of image content found on the page.
 * This value cannot be {@code null}.
 * @throws java.lang.IllegalStateException If the document/page is closed before invocation.
 */

@android.annotation.NonNull
public java.util.List<android.graphics.pdf.content.PdfPageImageContent> getImageContents() { throw new RuntimeException("Stub!"); }

/**
 * Gets the page index.
 *
 * @return The index.

 * Value is 0 or greater
 */

public int getIndex() { throw new RuntimeException("Stub!"); }

/**
 * Get the bounds and URLs of all the links on the given page.
 *
 * @return list of all links on the page.
 * This value cannot be {@code null}.
 * @throws java.lang.IllegalStateException If the document/page is closed before invocation.
 */

@android.annotation.NonNull
public java.util.List<android.graphics.pdf.content.PdfPageLinkContent> getLinkContents() { throw new RuntimeException("Stub!"); }

/**
 * Return list of {@link android.graphics.pdf.content.PdfPageTextContent PdfPageTextContent} found on the page, ordered left to right
 * and top to bottom. It contains all the content associated with text found on the page.
 * The list will be empty if there are no results found. Currently, localisation does
 * not have any impact on the order in which {@link android.graphics.pdf.content.PdfPageTextContent PdfPageTextContent} is returned.
 *
 * @return list of text content found on the page.
 * This value cannot be {@code null}.
 * @throws java.lang.IllegalStateException If the document/page is closed before invocation.
 */

@android.annotation.NonNull
public java.util.List<android.graphics.pdf.content.PdfPageTextContent> getTextContents() { throw new RuntimeException("Stub!"); }

/**
 * Returns the width of the given {@link android.graphics.pdf.PdfRendererPreV.Page Page} object in points (1/72"). It is not
 * guaranteed that all pages will have the same width and the viewport should be resized to
 * the given page width.
 *
 * @return width of the given page
 * Value is 0 or greater
 * @throws java.lang.IllegalStateException If the document/page is closed before invocation.
 */

public int getWidth() { throw new RuntimeException("Stub!"); }

/**
 * Renders a page to a bitmap. In case of default zoom, the {@link android.graphics.Bitmap Bitmap} dimensions will
 * be equal to the page dimensions. In this case, {@link android.graphics.Rect Rect} parameter can be null.
 *
 * <p>In case of zoom, the {@link android.graphics.Rect Rect} parameter needs to be specified which represents
 * the offset from top and left for tile generation purposes. In this case, the
 * {@link android.graphics.Bitmap Bitmap} dimensions should be equal to the tile dimensions.
 * <p>
 * <strong>Note:</strong> The method will take care of closing the bitmap. Should be
 * invoked
 * on the {@link android.annotation.WorkerThread} as it is long-running task.
 *
 * @param destination Destination bitmap to write to.
 * This value cannot be {@code null}.
 * @param destClip    If null, default zoom is applied. In case the value is non-null, the
 *                    value specifies the top top-left corner of the tile.
 * @param transform   Applied to scale the bitmap up/down from default 1/72 points.
 * This value may be {@code null}.
 * @param params      Render params for the changing display mode and/or annotations.
 * This value cannot be {@code null}.
 * @throws java.lang.IllegalStateException If the document/page is closed before invocation.
 */

public void render(@android.annotation.NonNull android.graphics.Bitmap destination, @android.annotation.Nullable android.graphics.Rect destClip, @android.annotation.Nullable android.graphics.Matrix transform, @android.annotation.NonNull android.graphics.pdf.RenderParams params) { throw new RuntimeException("Stub!"); }

/**
 * Search for the given string on the page and returns the bounds of all the matches. The
 * list will be empty if there are no matches on the given page. If this function was
 * invoked previously for any page, it will wait for that operation to
 * complete before this operation is started.
 * <p>
 * <strong>Note:</strong> Should be invoked on the {@link android.annotation.WorkerThread}
 * as it is long-running task.
 *
 * @param query plain search string for querying the document
 * This value cannot be {@code null}.
 * @return List of {@link android.graphics.pdf.models.PageMatchBounds PageMatchBounds} representing the bounds of each match on the
 * page.
 * This value cannot be {@code null}.
 * @throws java.lang.IllegalStateException If the document/page is closed before invocation.
 */

@android.annotation.NonNull
public java.util.List<android.graphics.pdf.models.PageMatchBounds> searchText(@android.annotation.NonNull java.lang.String query) { throw new RuntimeException("Stub!"); }

/**
 * Return a {@link android.graphics.pdf.models.selection.PageSelection PageSelection} which represents the selected content that spans between
 * the two boundaries. The boundaries can be either exactly defined with text indexes, or
 * approximately defined with points on the page. The resulting selection will also be
 * exactly defined with both indexes and points. If the start and stop boundary are both at
 * the same point, selects the word at that point. In case the selection from the given
 * boundaries result in an empty space, then the method returns {@code null}. The start and
 * stop {@link android.graphics.pdf.models.selection.SelectionBoundary SelectionBoundary} in {@link android.graphics.pdf.models.selection.PageSelection PageSelection} resolves to the "nearest" index
 * when returned.
 * <p>
 * <strong>Note:</strong> Should be invoked on a {@link android.annotation.WorkerThread}
 * as it is long-running task.
 *
 * @param start boundary where the selection starts (inclusive)
 * This value cannot be {@code null}.
 * @param stop  boundary where the selection stops (exclusive)
 * This value cannot be {@code null}.
 * @return collection of the selected content for text, images, etc.
 * @throws java.lang.IllegalStateException If the document/page is closed before invocation.
 */

@android.annotation.Nullable
public android.graphics.pdf.models.selection.PageSelection selectContent(@android.annotation.NonNull android.graphics.pdf.models.selection.SelectionBoundary start, @android.annotation.NonNull android.graphics.pdf.models.selection.SelectionBoundary stop) { throw new RuntimeException("Stub!"); }
}

}

