/*
 * Copyright (C) 2024 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */


package android.widget.photopicker;

import android.content.Context;
import android.hardware.display.DisplayManager;
import android.view.AttachedSurfaceControl;
import java.util.concurrent.Executor;

/**
 * This interface provides an api that callers can use to get a session of embedded PhotoPicker
 * ({@link android.widget.photopicker.EmbeddedPhotoPickerSession EmbeddedPhotoPickerSession}).
 *
 * <p> Callers can get instance of this class using
 * {@link android.widget.photopicker.EmbeddedPhotoPickerProviderFactory#create(android.content.Context) EmbeddedPhotoPickerProviderFactory.create(Context)}.
 *
 * <p> Under the hood, a service connection with photopicker is established by the implementation
 * of this api. To help establish this connection, a caller must include in their Manifest:
 * <pre>{@code
 * <queries>
 *   <intent>
 *       <action android:name="com.android.photopicker.core.embedded.EmbeddedService.BIND"/>
 *   </intent>
 * </queries>
 * }</pre>
 *
 * <p> When a session opens successfully, they would receive an instance of
 * {@link android.widget.photopicker.EmbeddedPhotoPickerSession EmbeddedPhotoPickerSession} and {@link android.view.SurfaceControlViewHost.SurfacePackage}
 * via the {@link android.widget.photopicker.EmbeddedPhotoPickerClient#onSessionOpened api}
 *
 * <p> Callers pass an instance of {@link android.widget.photopicker.EmbeddedPhotoPickerClient EmbeddedPhotoPickerClient} which is used by service to
 * notify about different events (like sessionError, uri granted/revoked etc) to them.
 * One-to-one relationship of client to session must be maintained by a caller i.e. they shouldn't
 * reuse same callback for more than one openSession requests.
 *
 * <p> The {@link android.widget.photopicker.EmbeddedPhotoPickerSession EmbeddedPhotoPickerSession} instance can be used to notify photopicker about
 * different events (like resize, configChange etc).
 *
 * <p> This api is supported on api versions Android U+.
 *
 * @see android.widget.photopicker.EmbeddedPhotoPickerClient
 * @see android.widget.photopicker.EmbeddedPhotoPickerSession
 * @see android.widget.photopicker.EmbeddedPhotoPickerProviderFactory
 *
 * todo(b/358513325): Move this to new package when its ready
 */

@SuppressWarnings({"unchecked", "deprecation", "all"})
public interface EmbeddedPhotoPickerProvider {

/**
 * Open a new session for displaying content with an initial size of
 * width x height pixels. {@link android.widget.photopicker.EmbeddedPhotoPickerClient EmbeddedPhotoPickerClient} will receive all incoming
 * communication from the PhotoPicker. All incoming calls to {@link android.widget.photopicker.EmbeddedPhotoPickerClient EmbeddedPhotoPickerClient}
 * will be made through the provided {@code clientExecutor}
 *
 * @param hostToken Token used for constructing {@link android.view.SurfaceControlViewHost}.
 *                  Use {@link android.view.AttachedSurfaceControl#getInputTransferToken() AttachedSurfaceControl.getInputTransferToken()} to
 *                  get token of attached
 *                  {@link android.view.SurfaceControlViewHost.SurfacePackage}.
 * @param displayId Application display id. Use
 *                  {@link android.hardware.display.DisplayManager#getDisplays() DisplayManager.getDisplays()} to get the id.
 * @param width width of the view, in pixels.
 * @param height height of the view, in pixels.
 * @param featureInfo {@link android.widget.photopicker.EmbeddedPhotoPickerFeatureInfo EmbeddedPhotoPickerFeatureInfo} object containing all
 *                     the required features for the given session.
 * @param clientExecutor {@link java.util.concurrent.Executor Executor} to invoke callbacks.
 * @param callback {@link android.widget.photopicker.EmbeddedPhotoPickerClient EmbeddedPhotoPickerClient} object to receive callbacks
 *                  from photopicker.
 */

public void openSession(@android.annotation.NonNull android.os.IBinder hostToken, int displayId, int width, int height, @android.annotation.NonNull android.widget.photopicker.EmbeddedPhotoPickerFeatureInfo featureInfo, @android.annotation.NonNull java.util.concurrent.Executor clientExecutor, @android.annotation.NonNull android.widget.photopicker.EmbeddedPhotoPickerClient callback);
}

