/*
 * Copyright (C) 2021 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */


package android.provider;

import android.os.Bundle;
import android.content.ContentResolver;
import android.database.Cursor;
import android.os.CancellationSignal;
import android.view.Surface;
import android.graphics.Point;

/**
 * Base class for a cloud media provider. A cloud media provider offers read-only access to durable
 * media files, specifically photos and videos stored on a local disk, or files in a cloud storage
 * service. To create a cloud media provider, extend this class, implement the abstract methods,
 * and add it to your manifest like this:
 *
 * <pre class="prettyprint">&lt;manifest&gt;
 *    ...
 *    &lt;application&gt;
 *        ...
 *        &lt;provider
 *            android:name="com.example.MyCloudProvider"
 *            android:authorities="com.example.mycloudprovider"
 *            android:exported="true"
 *            android:permission="com.android.providers.media.permission.MANAGE_CLOUD_MEDIA_PROVIDERS"
 *            &lt;intent-filter&gt;
 *                &lt;action android:name="android.content.action.CLOUD_MEDIA_PROVIDER" /&gt;
 *            &lt;/intent-filter&gt;
 *        &lt;/provider&gt;
 *        ...
 *    &lt;/application&gt;
 *&lt;/manifest&gt;</pre>
 * <p>
 * When defining your provider, you must protect it with the
 * {@link android.provider.CloudMediaProviderContract#MANAGE_CLOUD_MEDIA_PROVIDERS_PERMISSION CloudMediaProviderContract#MANAGE_CLOUD_MEDIA_PROVIDERS_PERMISSION}, which is a permission
 * only the system can obtain, trying to define an unprotected {@link android.provider.CloudMediaProvider CloudMediaProvider} will
 * result in a {@link java.lang.SecurityException SecurityException}.
 * <p>
 * Applications cannot use a cloud media provider directly; they must go through
 * {@link android.provider.MediaStore#ACTION_PICK_IMAGES MediaStore#ACTION_PICK_IMAGES} which requires a user to actively navigate and select
 * media items. When a user selects a media item through that UI, the system issues narrow URI
 * permission grants to the requesting application.
 * <h3>Media items</h3>
 * <p>
 * A media item must be an openable stream (with a specific MIME type). Media items can belong to
 * zero or more albums. Albums cannot contain other albums.
 * <p>
 * Each item under a provider is uniquely referenced by its media or album id, which must not
 * change which must be unique across all collection IDs as returned by
 * {@link #onGetMediaCollectionInfo}.
 *
 * @see android.provider.MediaStore#ACTION_PICK_IMAGES
 */

@SuppressWarnings({"unchecked", "deprecation", "all"})
public abstract class CloudMediaProvider extends android.content.ContentProvider {

public CloudMediaProvider() { throw new RuntimeException("Stub!"); }

/**
 * Implementation is provided by the parent class. Cannot be overridden.
 
 * @param context This value cannot be {@code null}.

 * @param info This value cannot be {@code null}.
 */

public final void attachInfo(@android.annotation.NonNull android.content.Context context, @android.annotation.NonNull android.content.pm.ProviderInfo info) { throw new RuntimeException("Stub!"); }

/**
 * Returns metadata about the media collection itself.
 * <p>
 * This is useful for the OS to determine if its cache of media items in the collection is
 * still valid and if a full or incremental sync is required with {@link #onQueryMedia}.
 * <p>
 * This method might be called by the OS frequently and is performance critical, hence it should
 * avoid long running operations.
 * <p>
 * If the provider handled any filters in {@code extras}, it must add the key to the
 * {@link android.content.ContentResolver#EXTRA_HONORED_ARGS ContentResolver#EXTRA_HONORED_ARGS} as part of the returned {@link android.os.Bundle Bundle}.
 *
 * @param extras containing keys to filter result:
 * <ul>
 * <li> {@link android.provider.CloudMediaProviderContract#EXTRA_ALBUM_ID CloudMediaProviderContract#EXTRA_ALBUM_ID}
 * </ul>
 *
 * This value cannot be {@code null}.
 * @return {@link android.os.Bundle Bundle} containing {@link android.provider.CloudMediaProviderContract.MediaCollectionInfo CloudMediaProviderContract.MediaCollectionInfo}
 * <ul>
 * <li> {@link android.provider.CloudMediaProviderContract.MediaCollectionInfo#MEDIA_COLLECTION_ID CloudMediaProviderContract.MediaCollectionInfo#MEDIA_COLLECTION_ID}
 * <li> {@link android.provider.CloudMediaProviderContract.MediaCollectionInfo#LAST_MEDIA_SYNC_GENERATION CloudMediaProviderContract.MediaCollectionInfo#LAST_MEDIA_SYNC_GENERATION}
 * <li> {@link android.provider.CloudMediaProviderContract.MediaCollectionInfo#ACCOUNT_NAME CloudMediaProviderContract.MediaCollectionInfo#ACCOUNT_NAME}
 * <li> {@link android.provider.CloudMediaProviderContract.MediaCollectionInfo#ACCOUNT_CONFIGURATION_INTENT CloudMediaProviderContract.MediaCollectionInfo#ACCOUNT_CONFIGURATION_INTENT}
 * </ul>
 
 * This value cannot be {@code null}.
 */

@android.annotation.NonNull
public abstract android.os.Bundle onGetMediaCollectionInfo(@android.annotation.NonNull android.os.Bundle extras);

/**
 * Returns a cursor representing all media items in the media collection optionally filtered by
 * {@code extras} and sorted in reverse chronological order of
 * {@link android.provider.CloudMediaProviderContract.MediaColumns#DATE_TAKEN_MILLIS CloudMediaProviderContract.MediaColumns#DATE_TAKEN_MILLIS}, i.e. most recent items
 * first.
 * <p>
 * The cloud media provider must set the
 * {@link android.provider.CloudMediaProviderContract#EXTRA_MEDIA_COLLECTION_ID CloudMediaProviderContract#EXTRA_MEDIA_COLLECTION_ID} as part of the returned
 * {@link android.database.Cursor#setExtras Cursor#setExtras} {@link android.os.Bundle Bundle}. Not setting this is an error and invalidates the
 * returned {@link android.database.Cursor Cursor}.
 * <p>
 * If the cloud media provider handled any filters in {@code extras}, it must add the key to
 * the {@link android.content.ContentResolver#EXTRA_HONORED_ARGS ContentResolver#EXTRA_HONORED_ARGS} as part of the returned
 * {@link android.database.Cursor#setExtras Cursor#setExtras} {@link android.os.Bundle Bundle}.
 *
 * @param extras containing keys to filter media items:
 * <ul>
 * <li> {@link android.provider.CloudMediaProviderContract#EXTRA_SYNC_GENERATION CloudMediaProviderContract#EXTRA_SYNC_GENERATION}
 * <li> {@link android.provider.CloudMediaProviderContract#EXTRA_PAGE_TOKEN CloudMediaProviderContract#EXTRA_PAGE_TOKEN}
 * <li> {@link android.provider.CloudMediaProviderContract#EXTRA_ALBUM_ID CloudMediaProviderContract#EXTRA_ALBUM_ID}
 * <li> {@link android.provider.CloudMediaProviderContract#EXTRA_PAGE_SIZE CloudMediaProviderContract#EXTRA_PAGE_SIZE}
 * </ul>
 * This value cannot be {@code null}.
 * @return cursor representing media items containing all
 * {@link android.provider.CloudMediaProviderContract.MediaColumns CloudMediaProviderContract.MediaColumns} columns
 
 * This value cannot be {@code null}.
 */

@android.annotation.NonNull
public abstract android.database.Cursor onQueryMedia(@android.annotation.NonNull android.os.Bundle extras);

/**
 * Returns a {@link android.database.Cursor Cursor} representing all deleted media items in the entire media collection
 * within the current provider version as returned by {@link #onGetMediaCollectionInfo}. These
 * items can be optionally filtered by {@code extras}.
 * <p>
 * The cloud media provider must set the
 * {@link android.provider.CloudMediaProviderContract#EXTRA_MEDIA_COLLECTION_ID CloudMediaProviderContract#EXTRA_MEDIA_COLLECTION_ID} as part of the returned
 * {@link android.database.Cursor#setExtras Cursor#setExtras} {@link android.os.Bundle Bundle}. Not setting this is an error and invalidates the
 * returned {@link android.database.Cursor Cursor}.
 * <p>
 * If the provider handled any filters in {@code extras}, it must add the key to
 * the {@link android.content.ContentResolver#EXTRA_HONORED_ARGS ContentResolver#EXTRA_HONORED_ARGS} as part of the returned
 * {@link android.database.Cursor#setExtras Cursor#setExtras} {@link android.os.Bundle Bundle}.
 *
 * @param extras containing keys to filter deleted media items:
 * <ul>
 * <li> {@link android.provider.CloudMediaProviderContract#EXTRA_SYNC_GENERATION CloudMediaProviderContract#EXTRA_SYNC_GENERATION}
 * <li> {@link android.provider.CloudMediaProviderContract#EXTRA_PAGE_TOKEN CloudMediaProviderContract#EXTRA_PAGE_TOKEN}
 * </ul>
 * This value cannot be {@code null}.
 * @return cursor representing deleted media items containing just the
 * {@link android.provider.CloudMediaProviderContract.MediaColumns#ID CloudMediaProviderContract.MediaColumns#ID} column
 
 * This value cannot be {@code null}.
 */

@android.annotation.NonNull
public abstract android.database.Cursor onQueryDeletedMedia(@android.annotation.NonNull android.os.Bundle extras);

/**
 * Returns a cursor representing all album items in the media collection optionally filtered
 * by {@code extras} and sorted in reverse chronological order of
 * {@link android.provider.CloudMediaProviderContract.AlbumColumns#DATE_TAKEN_MILLIS CloudMediaProviderContract.AlbumColumns#DATE_TAKEN_MILLIS}, i.e. most recent items
 * first.
 * <p>
 * The cloud media provider must set the
 * {@link android.provider.CloudMediaProviderContract#EXTRA_MEDIA_COLLECTION_ID CloudMediaProviderContract#EXTRA_MEDIA_COLLECTION_ID} as part of the returned
 * {@link android.database.Cursor#setExtras Cursor#setExtras} {@link android.os.Bundle Bundle}. Not setting this is an error and invalidates the
 * returned {@link android.database.Cursor Cursor}.
 * <p>
 * If the provider handled any filters in {@code extras}, it must add the key to
 * the {@link android.content.ContentResolver#EXTRA_HONORED_ARGS ContentResolver#EXTRA_HONORED_ARGS} as part of the returned
 * {@link android.database.Cursor#setExtras Cursor#setExtras} {@link android.os.Bundle Bundle}.
 *
 * @param extras containing keys to filter album items:
 * <ul>
 * <li> {@link android.provider.CloudMediaProviderContract#EXTRA_SYNC_GENERATION CloudMediaProviderContract#EXTRA_SYNC_GENERATION}
 * <li> {@link android.provider.CloudMediaProviderContract#EXTRA_PAGE_TOKEN CloudMediaProviderContract#EXTRA_PAGE_TOKEN}
 * <li> {@link android.provider.CloudMediaProviderContract#EXTRA_PAGE_SIZE CloudMediaProviderContract#EXTRA_PAGE_SIZE}
 * </ul>
 * This value cannot be {@code null}.
 * @return cursor representing album items containing all
 * {@link android.provider.CloudMediaProviderContract.AlbumColumns CloudMediaProviderContract.AlbumColumns} columns
 
 * This value cannot be {@code null}.
 */

@android.annotation.NonNull
public android.database.Cursor onQueryAlbums(@android.annotation.NonNull android.os.Bundle extras) { throw new RuntimeException("Stub!"); }

/**
 * Returns a thumbnail of {@code size} for a media item identified by {@code mediaId}
 * <p>The cloud media provider should strictly return thumbnail in the original
 * {@link android.provider.CloudMediaProviderContract.MediaColumns#MIME_TYPE CloudMediaProviderContract.MediaColumns#MIME_TYPE} of the item.
 * <p>
 * This is expected to be a much lower resolution version than the item returned by
 * {@link #onOpenMedia}.
 * <p>
 * If you block while downloading content, you should periodically check
 * {@link android.os.CancellationSignal#isCanceled() CancellationSignal#isCanceled()} to abort abandoned open requests.
 *
 * @param mediaId the media item to return
 * This value cannot be {@code null}.
 * @param size the dimensions of the thumbnail to return. The returned file descriptor doesn't
 * have to match the {@code size} precisely because the OS will adjust the dimensions before
 * usage. Implementations can return close approximations especially if the approximation is
 * already locally on the device and doesn't require downloading from the cloud.
 * This value cannot be {@code null}.
 * @param extras to modify the way the fd is opened, e.g. for video files we may request a
 * thumbnail image instead of a video with
 * {@link android.provider.CloudMediaProviderContract#EXTRA_PREVIEW_THUMBNAIL CloudMediaProviderContract#EXTRA_PREVIEW_THUMBNAIL}
 * This value may be {@code null}.
 * @param signal used by the OS to signal if the request should be cancelled
 * This value may be {@code null}.
 * @return read-only file descriptor for accessing the thumbnail for the media file
 *
 * This value cannot be {@code null}.
 * @see #onOpenMedia
 * @see android.provider.CloudMediaProviderContract#EXTRA_PREVIEW_THUMBNAIL
 */

@android.annotation.NonNull
public abstract android.content.res.AssetFileDescriptor onOpenPreview(@android.annotation.NonNull java.lang.String mediaId, @android.annotation.NonNull android.graphics.Point size, @android.annotation.Nullable android.os.Bundle extras, @android.annotation.Nullable android.os.CancellationSignal signal) throws java.io.FileNotFoundException;

/**
 * Returns the full size media item identified by {@code mediaId}.
 * <p>
 * If you block while downloading content, you should periodically check
 * {@link android.os.CancellationSignal#isCanceled() CancellationSignal#isCanceled()} to abort abandoned open requests.
 *
 * @param mediaId the media item to return
 * This value cannot be {@code null}.
 * @param extras to modify the way the fd is opened, there's none at the moment, but some
 * might be implemented in the future
 * This value may be {@code null}.
 * @param signal used by the OS to signal if the request should be cancelled
 * This value may be {@code null}.
 * @return read-only file descriptor for accessing the media file
 *
 * This value cannot be {@code null}.
 * @see #onOpenPreview
 */

@android.annotation.NonNull
public abstract android.os.ParcelFileDescriptor onOpenMedia(@android.annotation.NonNull java.lang.String mediaId, @android.annotation.Nullable android.os.Bundle extras, @android.annotation.Nullable android.os.CancellationSignal signal) throws java.io.FileNotFoundException;

/**
 * Returns a {@link android.provider.CloudMediaProvider.CloudMediaSurfaceController CloudMediaSurfaceController} used for rendering the preview of media items,
 * or null if preview rendering is not supported.
 *
 * @param config containing configuration parameters for {@link android.provider.CloudMediaProvider.CloudMediaSurfaceController CloudMediaSurfaceController}
 * <ul>
 * <li> {@link android.provider.CloudMediaProviderContract#EXTRA_LOOPING_PLAYBACK_ENABLED CloudMediaProviderContract#EXTRA_LOOPING_PLAYBACK_ENABLED}
 * <li> {@link android.provider.CloudMediaProviderContract#EXTRA_SURFACE_CONTROLLER_AUDIO_MUTE_ENABLED CloudMediaProviderContract#EXTRA_SURFACE_CONTROLLER_AUDIO_MUTE_ENABLED}
 * </ul>
 * This value cannot be {@code null}.
 * @param callback {@link android.provider.CloudMediaProvider.CloudMediaSurfaceStateChangedCallback CloudMediaSurfaceStateChangedCallback} to send state updates for
 *                 {@link android.view.Surface Surface} to picker launched via {@link android.provider.MediaStore#ACTION_PICK_IMAGES MediaStore#ACTION_PICK_IMAGES}

 * This value cannot be {@code null}.
 */

@android.annotation.Nullable
public android.provider.CloudMediaProvider.CloudMediaSurfaceController onCreateCloudMediaSurfaceController(@android.annotation.NonNull android.os.Bundle config, @android.annotation.NonNull android.provider.CloudMediaProvider.CloudMediaSurfaceStateChangedCallback callback) { throw new RuntimeException("Stub!"); }

/**
 * Implementation is provided by the parent class. Cannot be overridden.
 
 * @param method This value cannot be {@code null}.
 * @param arg This value may be {@code null}.
 * @param extras This value may be {@code null}.
 * @return This value cannot be {@code null}.
 */

@android.annotation.NonNull
public final android.os.Bundle call(@android.annotation.NonNull java.lang.String method, @android.annotation.Nullable java.lang.String arg, @android.annotation.Nullable android.os.Bundle extras) { throw new RuntimeException("Stub!"); }

/**
 * Implementation is provided by the parent class. Cannot be overridden.
 *
 * @see #onOpenMedia
 
 * @param uri This value cannot be {@code null}.
 * @param mode This value cannot be {@code null}.
 * @return This value cannot be {@code null}.
 */

@android.annotation.NonNull
public final android.os.ParcelFileDescriptor openFile(@android.annotation.NonNull android.net.Uri uri, @android.annotation.NonNull java.lang.String mode) throws java.io.FileNotFoundException { throw new RuntimeException("Stub!"); }

/**
 * Implementation is provided by the parent class. Cannot be overridden.
 *
 * @see #onOpenMedia
 
 * @param uri This value cannot be {@code null}.
 * @param mode This value cannot be {@code null}.
 * @param signal This value may be {@code null}.
 * @return This value cannot be {@code null}.
 */

@android.annotation.NonNull
public final android.os.ParcelFileDescriptor openFile(@android.annotation.NonNull android.net.Uri uri, @android.annotation.NonNull java.lang.String mode, @android.annotation.Nullable android.os.CancellationSignal signal) throws java.io.FileNotFoundException { throw new RuntimeException("Stub!"); }

/**
 * Implementation is provided by the parent class. Cannot be overridden.
 *
 * @see #onOpenPreview
 * @see #onOpenMedia
 
 * @param uri This value cannot be {@code null}.
 * @param mimeTypeFilter This value cannot be {@code null}.
 * @param opts This value may be {@code null}.
 * @return This value cannot be {@code null}.
 */

@android.annotation.NonNull
public final android.content.res.AssetFileDescriptor openTypedAssetFile(@android.annotation.NonNull android.net.Uri uri, @android.annotation.NonNull java.lang.String mimeTypeFilter, @android.annotation.Nullable android.os.Bundle opts) throws java.io.FileNotFoundException { throw new RuntimeException("Stub!"); }

/**
 * Implementation is provided by the parent class. Cannot be overridden.
 *
 * @see #onOpenPreview
 * @see #onOpenMedia
 
 * @param uri This value cannot be {@code null}.
 * @param mimeTypeFilter This value cannot be {@code null}.
 * @param opts This value may be {@code null}.
 * @param signal This value may be {@code null}.
 * @return This value cannot be {@code null}.
 */

@android.annotation.NonNull
public final android.content.res.AssetFileDescriptor openTypedAssetFile(@android.annotation.NonNull android.net.Uri uri, @android.annotation.NonNull java.lang.String mimeTypeFilter, @android.annotation.Nullable android.os.Bundle opts, @android.annotation.Nullable android.os.CancellationSignal signal) throws java.io.FileNotFoundException { throw new RuntimeException("Stub!"); }

/**
 * Implementation is provided by the parent class. Cannot be overridden.
 *
 * @see #onQueryMedia
 * @see #onQueryDeletedMedia
 * @see #onQueryAlbums
 
 * @param uri This value cannot be {@code null}.
 * @param projection This value may be {@code null}.
 * @param queryArgs This value may be {@code null}.
 * @param cancellationSignal This value may be {@code null}.
 * @return This value cannot be {@code null}.
 */

@android.annotation.NonNull
public final android.database.Cursor query(@android.annotation.NonNull android.net.Uri uri, @android.annotation.Nullable java.lang.String[] projection, @android.annotation.Nullable android.os.Bundle queryArgs, @android.annotation.Nullable android.os.CancellationSignal cancellationSignal) { throw new RuntimeException("Stub!"); }

/**
 * Implementation is provided by the parent class. Throws by default, and
 * cannot be overridden.
 
 * @param uri This value cannot be {@code null}.
 * @return This value cannot be {@code null}.
 */

@android.annotation.NonNull
public final java.lang.String getType(@android.annotation.NonNull android.net.Uri uri) { throw new RuntimeException("Stub!"); }

/**
 * Implementation is provided by the parent class. Throws by default, and
 * cannot be overridden.
 
 * @param uri This value cannot be {@code null}.
 * @return This value cannot be {@code null}.
 */

@android.annotation.NonNull
public final android.net.Uri canonicalize(@android.annotation.NonNull android.net.Uri uri) { throw new RuntimeException("Stub!"); }

/**
 * Implementation is provided by the parent class. Throws by default, and
 * cannot be overridden.
 
 * @param uri This value cannot be {@code null}.
 * @param projection This value may be {@code null}.
 * @param selection This value may be {@code null}.
 * @param selectionArgs This value may be {@code null}.
 * @param sortOrder This value may be {@code null}.
 * @return This value cannot be {@code null}.
 */

@android.annotation.NonNull
public final android.database.Cursor query(@android.annotation.NonNull android.net.Uri uri, @android.annotation.Nullable java.lang.String[] projection, @android.annotation.Nullable java.lang.String selection, @android.annotation.Nullable java.lang.String[] selectionArgs, @android.annotation.Nullable java.lang.String sortOrder) { throw new RuntimeException("Stub!"); }

/**
 * Implementation is provided by the parent class. Throws by default, and
 * cannot be overridden.
 
 * @param uri This value cannot be {@code null}.
 * @param projection This value may be {@code null}.
 * @param selection This value may be {@code null}.
 * @param selectionArgs This value may be {@code null}.
 * @param sortOrder This value may be {@code null}.
 * @param cancellationSignal This value may be {@code null}.
 * @return This value cannot be {@code null}.
 */

@android.annotation.NonNull
public final android.database.Cursor query(@android.annotation.NonNull android.net.Uri uri, @android.annotation.Nullable java.lang.String[] projection, @android.annotation.Nullable java.lang.String selection, @android.annotation.Nullable java.lang.String[] selectionArgs, @android.annotation.Nullable java.lang.String sortOrder, @android.annotation.Nullable android.os.CancellationSignal cancellationSignal) { throw new RuntimeException("Stub!"); }

/**
 * Implementation is provided by the parent class. Throws by default, and
 * cannot be overridden.
 
 * @param uri This value cannot be {@code null}.
 * @param values This value cannot be {@code null}.
 * @return This value cannot be {@code null}.
 */

@android.annotation.NonNull
public final android.net.Uri insert(@android.annotation.NonNull android.net.Uri uri, @android.annotation.NonNull android.content.ContentValues values) { throw new RuntimeException("Stub!"); }

/**
 * Implementation is provided by the parent class. Throws by default, and
 * cannot be overridden.

 * @param uri This value cannot be {@code null}.
 
 * @param selection This value may be {@code null}.

 * @param selectionArgs This value may be {@code null}.
 */

public final int delete(@android.annotation.NonNull android.net.Uri uri, @android.annotation.Nullable java.lang.String selection, @android.annotation.Nullable java.lang.String[] selectionArgs) { throw new RuntimeException("Stub!"); }

/**
 * Implementation is provided by the parent class. Throws by default, and
 * cannot be overridden.
 
 * @param uri This value cannot be {@code null}.

 * @param values This value cannot be {@code null}.
 
 * @param selection This value may be {@code null}.

 * @param selectionArgs This value may be {@code null}.
 */

public final int update(@android.annotation.NonNull android.net.Uri uri, @android.annotation.NonNull android.content.ContentValues values, @android.annotation.Nullable java.lang.String selection, @android.annotation.Nullable java.lang.String[] selectionArgs) { throw new RuntimeException("Stub!"); }
/**
 * Manages rendering the preview of media items on given instances of {@link android.view.Surface Surface}.
 *
 * <p>The methods of this class are meant to be asynchronous, and should not block by performing
 * any heavy operation.
 * <p>Note that a single CloudMediaSurfaceController instance would be responsible for
 * rendering multiple media items associated with multiple surfaces.
 */

@SuppressWarnings({"unchecked", "deprecation", "all"})
public abstract static class CloudMediaSurfaceController {

public CloudMediaSurfaceController() { throw new RuntimeException("Stub!"); }

/**
 * Creates any player resource(s) needed for rendering.
 */

public abstract void onPlayerCreate();

/**
 * Releases any player resource(s) used for rendering.
 */

public abstract void onPlayerRelease();

/**
 * Indicates creation of the given {@link android.view.Surface Surface} with given {@code surfaceId} for
 * rendering the preview of a media item with given {@code mediaId}.
 *
 * <p>This is called immediately after the surface is first created. Implementations of this
 * should start up whatever rendering code they desire.
 * <p>Note that the given media item remains associated with the given surface id till the
 * {@link android.view.Surface Surface} is destroyed.
 *
 * @param surfaceId id which uniquely identifies the {@link android.view.Surface Surface} for rendering
 * @param surface instance of the {@link android.view.Surface Surface} on which the media item should be rendered
 * This value cannot be {@code null}.
 * @param mediaId id which uniquely identifies the media to be rendered
 *
 * This value cannot be {@code null}.
 * @see android.view.SurfaceHolder.Callback#surfaceCreated(SurfaceHolder)
 */

public abstract void onSurfaceCreated(int surfaceId, @android.annotation.NonNull android.view.Surface surface, @android.annotation.NonNull java.lang.String mediaId);

/**
 * Indicates structural changes (format or size) in the {@link android.view.Surface Surface} for rendering.
 *
 * <p>This method is always called at least once, after {@link #onSurfaceCreated}.
 *
 * @param surfaceId id which uniquely identifies the {@link android.view.Surface Surface} for rendering
 * @param format the new {@link android.graphics.PixelFormat PixelFormat} of the surface
 * @param width the new width of the {@link android.view.Surface Surface}
 * @param height the new height of the {@link android.view.Surface Surface}
 *
 * @see android.view.SurfaceHolder.Callback#surfaceChanged(SurfaceHolder, int, int, int)
 */

public abstract void onSurfaceChanged(int surfaceId, int format, int width, int height);

/**
 * Indicates destruction of a {@link android.view.Surface Surface} with given {@code surfaceId}.
 *
 * <p>This is called immediately before a surface is being destroyed. After returning from
 * this call, you should no longer try to access this surface.
 *
 * @param surfaceId id which uniquely identifies the {@link android.view.Surface Surface} for rendering
 *
 * @see android.view.SurfaceHolder.Callback#surfaceDestroyed(SurfaceHolder)
 */

public abstract void onSurfaceDestroyed(int surfaceId);

/**
 * Start playing the preview of the media associated with the given surface id. If
 * playback had previously been paused, playback will continue from where it was paused.
 * If playback had been stopped, or never started before, playback will start at the
 * beginning.
 *
 * @param surfaceId id which uniquely identifies the {@link android.view.Surface Surface} for rendering
 */

public abstract void onMediaPlay(int surfaceId);

/**
 * Pauses the playback of the media associated with the given surface id.
 *
 * @param surfaceId id which uniquely identifies the {@link android.view.Surface Surface} for rendering
 */

public abstract void onMediaPause(int surfaceId);

/**
 * Seeks the media associated with the given surface id to specified timestamp.
 *
 * @param surfaceId id which uniquely identifies the {@link android.view.Surface Surface} for rendering
 * @param timestampMillis the timestamp in milliseconds from the start to seek to

 * Value is a non-negative duration in milliseconds.
 */

public abstract void onMediaSeekTo(int surfaceId, long timestampMillis);

/**
 * Changes the configuration parameters for the CloudMediaSurfaceController.
 *
 * @param config the updated config to change to. This can include config changes for the
 * following:
 * <ul>
 * <li> {@link android.provider.CloudMediaProviderContract#EXTRA_LOOPING_PLAYBACK_ENABLED CloudMediaProviderContract#EXTRA_LOOPING_PLAYBACK_ENABLED}
 * <li> {@link android.provider.CloudMediaProviderContract#EXTRA_SURFACE_CONTROLLER_AUDIO_MUTE_ENABLED CloudMediaProviderContract#EXTRA_SURFACE_CONTROLLER_AUDIO_MUTE_ENABLED}
 * </ul>

 * This value cannot be {@code null}.
 */

public abstract void onConfigChange(@android.annotation.NonNull android.os.Bundle config);

/**
 * Indicates destruction of this CloudMediaSurfaceController object.
 *
 * <p>This CloudMediaSurfaceController object should no longer be in use after this method
 * has been called.
 *
 * <p>Note that it is possible for this method to be called directly without
 * {@link #onPlayerRelease} being called, hence you should release any resources associated
 * with this CloudMediaSurfaceController object, or perform any cleanup required in this
 * method.
 */

public abstract void onDestroy();
}

/**
 * This class is used by {@link android.provider.CloudMediaProvider CloudMediaProvider} to send {@link android.view.Surface Surface} state updates to
 * picker launched via {@link android.provider.MediaStore#ACTION_PICK_IMAGES MediaStore#ACTION_PICK_IMAGES}.
 *
 * @see android.provider.MediaStore#ACTION_PICK_IMAGES
 */

@SuppressWarnings({"unchecked", "deprecation", "all"})
public static final class CloudMediaSurfaceStateChangedCallback {

CloudMediaSurfaceStateChangedCallback() { throw new RuntimeException("Stub!"); }

/**
 * This is called to notify playback state update for a {@link android.view.Surface Surface}
 * on the picker launched via {@link android.provider.MediaStore#ACTION_PICK_IMAGES MediaStore#ACTION_PICK_IMAGES}.
 *
 * @param surfaceId id which uniquely identifies a {@link android.view.Surface Surface}
 * @param playbackState playback state to notify picker about
 * Value is either <code>0</code> or a combination of {@link android.provider.CloudMediaProvider.CloudMediaSurfaceStateChangedCallback#PLAYBACK_STATE_BUFFERING}, {@link android.provider.CloudMediaProvider.CloudMediaSurfaceStateChangedCallback#PLAYBACK_STATE_READY}, {@link android.provider.CloudMediaProvider.CloudMediaSurfaceStateChangedCallback#PLAYBACK_STATE_STARTED}, {@link android.provider.CloudMediaProvider.CloudMediaSurfaceStateChangedCallback#PLAYBACK_STATE_PAUSED}, {@link android.provider.CloudMediaProvider.CloudMediaSurfaceStateChangedCallback#PLAYBACK_STATE_COMPLETED}, {@link android.provider.CloudMediaProvider.CloudMediaSurfaceStateChangedCallback#PLAYBACK_STATE_ERROR_RETRIABLE_FAILURE}, {@link android.provider.CloudMediaProvider.CloudMediaSurfaceStateChangedCallback#PLAYBACK_STATE_ERROR_PERMANENT_FAILURE}, and {@link android.provider.CloudMediaProvider.CloudMediaSurfaceStateChangedCallback#PLAYBACK_STATE_MEDIA_SIZE_CHANGED}
 * @param playbackStateInfo {@link android.os.Bundle Bundle} which may contain extra information about the
 *                          playback state, such as media size, progress/seek info or
 *                          details about errors.

 * This value may be {@code null}.
 */

public void setPlaybackState(int surfaceId, int playbackState, @android.annotation.Nullable android.os.Bundle playbackStateInfo) { throw new RuntimeException("Stub!"); }

/**
 * Constant to notify that the playback is buffering
 */

public static final int PLAYBACK_STATE_BUFFERING = 1; // 0x1

/**
 * Constant to notify that the playback has completed
 */

public static final int PLAYBACK_STATE_COMPLETED = 5; // 0x5

/**
 * Constant to notify that the playback has failed with a permanent error.
 */

public static final int PLAYBACK_STATE_ERROR_PERMANENT_FAILURE = 7; // 0x7

/**
 * Constant to notify that the playback has failed with a retriable error.
 */

public static final int PLAYBACK_STATE_ERROR_RETRIABLE_FAILURE = 6; // 0x6

/**
 * Constant to notify that the media size is first known or has changed.
 *
 * Pass the width and height of the media as a {@link android.graphics.Point Point} inside the {@link android.os.Bundle Bundle} with
 * {@link android.content.ContentResolver#EXTRA_SIZE ContentResolver#EXTRA_SIZE} as the key.
 *
 * @see android.provider.CloudMediaProvider.CloudMediaSurfaceStateChangedCallback#setPlaybackState(int, int, Bundle)
 * @see android.media.MediaPlayer.OnVideoSizeChangedListener#onVideoSizeChanged(MediaPlayer, int, int)
 */

public static final int PLAYBACK_STATE_MEDIA_SIZE_CHANGED = 8; // 0x8

/**
 * Constant to notify that the playback is paused.
 */

public static final int PLAYBACK_STATE_PAUSED = 4; // 0x4

/**
 * Constant to notify that the playback is ready to be played
 */

public static final int PLAYBACK_STATE_READY = 2; // 0x2

/**
 * Constant to notify that the playback has started
 */

public static final int PLAYBACK_STATE_STARTED = 3; // 0x3
}

}

