/*
 * Copyright (C) 2013 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */


package android.nfc.cardemulation;

import android.Manifest;
import android.app.Activity;
import android.app.role.RoleManager;
import android.content.ComponentName;
import android.content.Context;
import android.content.Intent;
import android.content.pm.PackageManager;
import android.nfc.NfcAdapter;
import android.os.Build;
import android.os.UserHandle;
import android.telephony.SubscriptionManager;
import java.util.List;
import java.util.concurrent.Executor;
import java.util.regex.Pattern;

/**
 * This class can be used to query the state of
 * NFC card emulation services.
 *
 * For a general introduction into NFC card emulation,
 * please read the <a href="{@docRoot}guide/topics/connectivity/nfc/hce.html">
 * NFC card emulation developer guide</a>.</p>
 *
 * <p class="note">Use of this class requires the
 * {@link android.content.pm.PackageManager#FEATURE_NFC_HOST_CARD_EMULATION PackageManager.FEATURE_NFC_HOST_CARD_EMULATION} to be present
 * on the device.
 */

@SuppressWarnings({"unchecked", "deprecation", "all"})
public final class CardEmulation {

CardEmulation() { throw new RuntimeException("Stub!"); }

/**
 * <p>
 * Returns whether the user has allowed AIDs registered in the
 * specified category to be handled by a service that is preferred
 * by the foreground application, instead of by a pre-configured default.
 *
 * Foreground applications can set such preferences using the
 * {@link #setPreferredService(android.app.Activity,android.content.ComponentName)} method.
 * <p class="note">
 * Starting with {@link android.os.Build.VERSION_CODES#VANILLA_ICE_CREAM Build.VERSION_CODES.VANILLA_ICE_CREAM}, this method will always
 * return true.
 *
 * @param category The category, e.g. {@link #CATEGORY_PAYMENT}
 * @return whether AIDs in the category can be handled by a service
 *         specified by the foreground app.
 */

public boolean categoryAllowsForegroundPreference(java.lang.String category) { throw new RuntimeException("Stub!"); }

/**
 * Retrieves the registered AIDs for the preferred payment service.
 *
 * <br>
 * Requires {@link android.Manifest.permission#NFC_PREFERRED_PAYMENT_INFO}
 * @return The list of AIDs registered for this category, or null if it couldn't be found.
 */

@android.annotation.Nullable
public java.util.List<java.lang.String> getAidsForPreferredPaymentService() { throw new RuntimeException("Stub!"); }

/**
 * Retrieves the currently registered AIDs for the specified
 * category for a service.
 *
 * <p>Note that this will only return AIDs that were dynamically
 * registered using {@link #registerAidsForService(android.content.ComponentName,java.lang.String,java.util.List)}
 * method. It will *not* return AIDs that were statically registered
 * in the manifest.
 *
 * @param service The component name of the service
 * @param category The category for which the AIDs were registered,
 *                 e.g. {@link #CATEGORY_PAYMENT}
 * @return The list of AIDs registered for this category, or null if it couldn't be found.
 */

public java.util.List<java.lang.String> getAidsForService(android.content.ComponentName service, java.lang.String category) { throw new RuntimeException("Stub!"); }

/**
 * Returns the system's default NFC subscription id.
 *
 * <p> For devices with multiple UICC/EUICC that is configured to be NFCEE, this returns the
 * default UICC NFCEE that will handle NFC offhost CE transactions </p>
 * <p> If the device has no UICC that can serve as NFCEE, this will return
 * {@link android.telephony.SubscriptionManager#INVALID_SUBSCRIPTION_ID SubscriptionManager.INVALID_SUBSCRIPTION_ID}.</p>
 *
 * <br>
 * Requires the {@link android.content.pm.PackageManager#FEATURE_TELEPHONY_SUBSCRIPTION PackageManager#FEATURE_TELEPHONY_SUBSCRIPTION} feature which can be detected using {@link android.content.pm.PackageManager#hasSystemFeature(String) PackageManager.hasSystemFeature(String)}.
 * @return the default NFC subscription Id if set,
 * {@link android.telephony.SubscriptionManager#INVALID_SUBSCRIPTION_ID SubscriptionManager.INVALID_SUBSCRIPTION_ID} otherwise.
 *
 * @throws java.lang.UnsupportedOperationException If the device does not have
 * {@link android.content.pm.PackageManager#FEATURE_TELEPHONY_SUBSCRIPTION PackageManager.FEATURE_TELEPHONY_SUBSCRIPTION}.
 */

public int getDefaultNfcSubscriptionId() { throw new RuntimeException("Stub!"); }

/**
 * Returns a user-visible description of the preferred payment service.
 *
 * <p class="note">
 * Starting with {@link android.os.Build.VERSION_CODES#VANILLA_ICE_CREAM Build.VERSION_CODES.VANILLA_ICE_CREAM}, the preferred payment service
 * no longer exists and is replaced by {@link android.app.role.RoleManager#ROLE_WALLET}. This
 * will return the description for one of the services registered by the role holder (if any).
 * If there are multiple services registered, it is unspecified which of those will be used
 * to obtain the service description here.
 *
 * <br>
 * Requires {@link android.Manifest.permission#NFC_PREFERRED_PAYMENT_INFO}
 * @return the preferred payment service description

 * This value may be {@code null}.
 */

@android.annotation.Nullable
public java.lang.CharSequence getDescriptionForPreferredPaymentService() { throw new RuntimeException("Stub!"); }

/**
 * Helper to get an instance of this class.
 *
 * @param adapter A reference to an NfcAdapter object.
 * @return
 */

public static synchronized android.nfc.cardemulation.CardEmulation getInstance(android.nfc.NfcAdapter adapter) { throw new RuntimeException("Stub!"); }

/**
 * Retrieves the route destination for the preferred payment service.
 *
 * <p class="note">
 * Starting with {@link android.os.Build.VERSION_CODES#VANILLA_ICE_CREAM Build.VERSION_CODES.VANILLA_ICE_CREAM}, the preferred payment service
 * no longer exists and is replaced by {@link android.app.role.RoleManager#ROLE_WALLET}. This
 * will return the route for one of the services registered by the role holder (if any). If
 * there are multiple services registered, it is unspecified which of those will be used to
 * determine the route.
 *
 * <br>
 * Requires {@link android.Manifest.permission#NFC_PREFERRED_PAYMENT_INFO}
 * @return The route destination secure element name of the preferred payment service.
 *         HCE payment: "Host"
 *         OffHost payment: 1. String with prefix SIM or prefix eSE string.
 *                             Ref: GSMA TS.26 - NFC Handset Requirements
 *                             TS26_NFC_REQ_069: For UICC, Secure Element Name SHALL be
 *                                               SIM[smartcard slot]
 *                                               (e.g. SIM/SIM1, SIM2… SIMn).
 *                             TS26_NFC_REQ_070: For embedded SE, Secure Element Name SHALL be
 *                                               eSE[number]
 *                                               (e.g. eSE/eSE1, eSE2, etc.).
 *                          2. "OffHost" if the payment service does not specify secure element
 *                             name.

 * This value may be {@code null}.
 */

@android.annotation.Nullable
public java.lang.String getRouteDestinationForPreferredPaymentService() { throw new RuntimeException("Stub!"); }

/**
 * Returns the service selection mode for the passed in category.
 * Valid return values are:
 * <p>{@link #SELECTION_MODE_PREFER_DEFAULT} the user has requested a default
 *    service for this category, which will be preferred.
 * <p>{@link #SELECTION_MODE_ALWAYS_ASK} the user has requested to be asked
 *    every time what service they would like to use in this category.
 * <p>{@link #SELECTION_MODE_ASK_IF_CONFLICT} the user will only be asked
 *    to pick a service if there is a conflict.
 *
 * <p class="note">
 * Starting with {@link android.os.Build.VERSION_CODES#VANILLA_ICE_CREAM Build.VERSION_CODES.VANILLA_ICE_CREAM}, the default service defined
 * by the holder of {@link android.app.role.RoleManager#ROLE_WALLET} and is category agnostic.
 *
 * @param category The category, for example {@link #CATEGORY_PAYMENT}
 * @return the selection mode for the passed in category
 */

public int getSelectionModeForCategory(java.lang.String category) { throw new RuntimeException("Stub!"); }

/**
 *
 * Allows an application to query whether a service is currently
 * the default handler for a specified ISO7816-4 Application ID.
 *
 * @param service The ComponentName of the service
 * @param aid The ISO7816-4 Application ID
 * @return whether the service is the default handler for the specified AID
 *
 * <p class="note">Requires the {@link android.Manifest.permission#NFC} permission.
 */

public boolean isDefaultServiceForAid(android.content.ComponentName service, java.lang.String aid) { throw new RuntimeException("Stub!"); }

/**
 * Allows an application to query whether a service is currently
 * the default service to handle a card emulation category.
 *
 * <p>Note that if {@link #getSelectionModeForCategory(java.lang.String)}
 * returns {@link #SELECTION_MODE_ALWAYS_ASK} or {@link #SELECTION_MODE_ASK_IF_CONFLICT},
 * this method will always return false. That is because in these
 * selection modes a default can't be set at the category level. For categories where
 * the selection mode is {@link #SELECTION_MODE_ALWAYS_ASK} or
 * {@link #SELECTION_MODE_ASK_IF_CONFLICT}, use
 * {@link #isDefaultServiceForAid(android.content.ComponentName,java.lang.String)} to determine whether a service
 * is the default for a specific AID.
 *
 * @param service The ComponentName of the service
 * @param category The category
 * @return whether service is currently the default service for the category.
 *
 * <p class="note">Requires the {@link android.Manifest.permission#NFC} permission.
 */

public boolean isDefaultServiceForCategory(android.content.ComponentName service, java.lang.String category) { throw new RuntimeException("Stub!"); }

/**
 * Is EUICC supported as a Secure Element EE which supports off host card emulation.
 *
 * @return true if the device supports EUICC for off host card emulation, false otherwise.
 */

public boolean isEuiccSupported() { throw new RuntimeException("Stub!"); }

/**
 * Registers a list of AIDs for a specific category for the
 * specified service.
 *
 * <p>If a list of AIDs for that category was previously
 * registered for this service (either statically
 * through the manifest, or dynamically by using this API),
 * that list of AIDs will be replaced with this one.
 *
 * <p>Note that you can only register AIDs for a service that
 * is running under the same UID as the caller of this API. Typically
 * this means you need to call this from the same
 * package as the service itself, though UIDs can also
 * be shared between packages using shared UIDs.
 *
 * @param service The component name of the service
 * @param category The category of AIDs to be registered
 * @param aids A list containing the AIDs to be registered
 * @return whether the registration was successful.
 */

public boolean registerAidsForService(android.content.ComponentName service, java.lang.String category, java.util.List<java.lang.String> aids) { throw new RuntimeException("Stub!"); }

/**
 * Register a listener for NFC Events.
 *
 * @param executor The Executor to run the call back with
 * This value cannot be {@code null}.
 * Callback and listener events are dispatched through this
 * {@link java.util.concurrent.Executor Executor}, providing an easy way to control which thread is
 * used. To dispatch events through the main thread of your
 * application, you can use
 * {@link android.content.Context#getMainExecutor() Context.getMainExecutor()}.
 * Otherwise, provide an {@link java.util.concurrent.Executor Executor} that dispatches to an appropriate thread.
 * @param listener The listener to register

 * This value cannot be {@code null}.
 */

public void registerNfcEventCallback(@android.annotation.NonNull java.util.concurrent.Executor executor, @android.annotation.NonNull android.nfc.cardemulation.CardEmulation.NfcEventCallback listener) { throw new RuntimeException("Stub!"); }

/**
 * Register a polling loop filter (PLF) for a HostApduService and indicate whether it should
 * auto-transact or not.  The PLF can be sequence of an
 * even number of at least 2 hexadecimal numbers (0-9, A-F or a-f), representing a series of
 * bytes. When non-standard polling loop frame matches this sequence exactly, it may be
 * delivered to {@link android.nfc.cardemulation.HostApduService#processPollingFrames(java.util.List) HostApduService.processPollingFrames(List)}.  If auto-transact
 * is set to true and this service is currently preferred or there are no other services
 * registered for this filter then observe mode will also be disabled.
 * @param service The HostApduService to register the filter for
 * This value cannot be {@code null}.
 * @param pollingLoopFilter The filter to register
 * This value cannot be {@code null}.
 * @param autoTransact true to have the NFC stack automatically disable observe mode and allow
 *         transactions to proceed when this filter matches, false otherwise
 * @return true if the filter was registered, false otherwise
 * @throws java.lang.IllegalArgumentException if the passed in string doesn't parse to at least one byte
 */

public boolean registerPollingLoopFilterForService(@android.annotation.NonNull android.content.ComponentName service, @android.annotation.NonNull java.lang.String pollingLoopFilter, boolean autoTransact) { throw new RuntimeException("Stub!"); }

/**
 * Register a polling loop pattern filter (PLPF) for a HostApduService and indicate whether it
 * should auto-transact or not. The pattern may include the characters 0-9 and A-F as well as
 * the regular expression operators `.`, `?` and `*`. When the beginning of anon-standard
 * polling loop frame matches this sequence exactly, it may be delivered to
 * {@link android.nfc.cardemulation.HostApduService#processPollingFrames(java.util.List) HostApduService.processPollingFrames(List)}. If auto-transact is set to true and this
 * service is currently preferred or there are no other services registered for this filter
 * then observe mode will also be disabled.
 * @param service The HostApduService to register the filter for
 * This value cannot be {@code null}.
 * @param pollingLoopPatternFilter The pattern filter to register, must to be compatible with
 *         {@link java.util.regex.Pattern#compile(String)} and only contain hexadecimal numbers
 *         and `.`, `?` and `*` operators
 * This value cannot be {@code null}.
 * @param autoTransact true to have the NFC stack automatically disable observe mode and allow
 *         transactions to proceed when this filter matches, false otherwise
 * @return true if the filter was registered, false otherwise
 * @throws java.lang.IllegalArgumentException if the filter containst elements other than hexadecimal
 *         numbers and `.`, `?` and `*` operators
 * @throws java.util.regex.PatternSyntaxException if the regex syntax is invalid
 */

public boolean registerPollingLoopPatternFilterForService(@android.annotation.NonNull android.content.ComponentName service, @android.annotation.NonNull java.lang.String pollingLoopPatternFilter, boolean autoTransact) { throw new RuntimeException("Stub!"); }

/**
 * Removes a previously registered list of AIDs for the specified category for the
 * service provided.
 *
 * <p>Note that this will only remove AIDs that were dynamically
 * registered using the {@link #registerAidsForService(android.content.ComponentName,java.lang.String,java.util.List)}
 * method. It will *not* remove AIDs that were statically registered in
 * the manifest. If dynamically registered AIDs are removed using
 * this method, and a statically registered AID group for the same category
 * exists in the manifest, the static AID group will become active again.
 *
 * @param service The component name of the service
 * @param category The category of the AIDs to be removed, e.g. {@link #CATEGORY_PAYMENT}
 * @return whether the group was successfully removed.
 */

public boolean removeAidsForService(android.content.ComponentName service, java.lang.String category) { throw new RuntimeException("Stub!"); }

/**
 * Unregister a polling loop filter (PLF) for a HostApduService. If the PLF had previously been
 * registered via {@link #registerPollingLoopFilterForService(android.content.ComponentName,java.lang.String,boolean)}
 * for this service it will be removed.
 * @param service The HostApduService to unregister the filter for
 * This value cannot be {@code null}.
 * @param pollingLoopFilter The filter to unregister
 * This value cannot be {@code null}.
 * @return true if the filter was removed, false otherwise
 * @throws java.lang.IllegalArgumentException if the passed in string doesn't parse to at least one byte
 */

public boolean removePollingLoopFilterForService(@android.annotation.NonNull android.content.ComponentName service, @android.annotation.NonNull java.lang.String pollingLoopFilter) { throw new RuntimeException("Stub!"); }

/**
 * Unregister a polling loop pattern filter (PLPF) for a HostApduService. If the PLF had
 * previously been registered via
 * {@link #registerPollingLoopFilterForService(android.content.ComponentName,java.lang.String,boolean)} for this
 * service it will be removed.
 * @param service The HostApduService to unregister the filter for
 * This value cannot be {@code null}.
 * @param pollingLoopPatternFilter The filter to unregister, must to be compatible with
 *         {@link java.util.regex.Pattern#compile(String)} and only contain hexadecimal numbers
 *         and`.`, `?` and `*` operators
 * This value cannot be {@code null}.
 * @return true if the filter was removed, false otherwise
 * @throws java.lang.IllegalArgumentException if the filter containst elements other than hexadecimal
 *         numbers and `.`, `?` and `*` operators
 * @throws java.util.regex.PatternSyntaxException if the regex syntax is invalid
 */

public boolean removePollingLoopPatternFilterForService(@android.annotation.NonNull android.content.ComponentName service, @android.annotation.NonNull java.lang.String pollingLoopPatternFilter) { throw new RuntimeException("Stub!"); }

/**
 * Sets the off-host Secure Element for the given service.
 *
 * <p>If off-host SE was initially set (either statically
 * through the manifest, or dynamically by using this API),
 * it will be replaced with this one. All AIDs registered by
 * this service will be re-routed to this Secure Element if
 * successful. AIDs that was statically assigned using manifest
 * will re-route to off-host SE that stated in manifest after NFC
 * toggle.
 *
 * <p>Note that you can only set off-host SE for a service that
 * is running under the same UID as the caller of this API. Typically
 * this means you need to call this from the same
 * package as the service itself, though UIDs can also
 * be shared between packages using shared UIDs.
 *
 * <p>Registeration will be successful only if the Secure Element
 * exists on the device.
 *
 * <br>
 * Requires {@link android.Manifest.permission#NFC}
 * @param service The component name of the service
 * This value cannot be {@code null}.
 * @param offHostSecureElement Secure Element to register the AID to. Only accept strings with
 *                             prefix SIM or prefix eSE.
 *                             Ref: GSMA TS.26 - NFC Handset Requirements
 *                             TS26_NFC_REQ_069: For UICC, Secure Element Name SHALL be
 *                                               SIM[smartcard slot]
 *                                               (e.g. SIM/SIM1, SIM2… SIMn).
 *                             TS26_NFC_REQ_070: For embedded SE, Secure Element Name SHALL be
 *                                               eSE[number]
 *                                               (e.g. eSE/eSE1, eSE2, etc.).
 * This value cannot be {@code null}.
 * @return whether the registration was successful.
 
 * This value cannot be {@code null}.
 */

@android.annotation.NonNull
public boolean setOffHostForService(@android.annotation.NonNull android.content.ComponentName service, @android.annotation.NonNull java.lang.String offHostSecureElement) { throw new RuntimeException("Stub!"); }

/**
 * Allows a foreground application to specify which card emulation service
 * should be preferred while a specific Activity is in the foreground.
 *
 * <p>The specified Activity must currently be in resumed state. A good
 * paradigm is to call this method in your {@link android.app.Activity#onResume Activity.onResume}, and to call
 * {@link #unsetPreferredService(android.app.Activity)} in your {@link android.app.Activity#onPause Activity.onPause}.
 *
 * <p>This method call will fail in two specific scenarios:
 * <ul>
 * <li> If the service registers one or more AIDs in the {@link #CATEGORY_PAYMENT}
 * category, but the user has indicated that foreground apps are not allowed
 * to override the default payment service.
 * <li> If the service registers one or more AIDs in the {@link #CATEGORY_OTHER}
 * category that are also handled by the default payment service, and the
 * user has indicated that foreground apps are not allowed to override the
 * default payment service.
 * </ul>
 *
 * <p> Use {@link #categoryAllowsForegroundPreference(java.lang.String)} to determine
 * whether foreground apps can override the default payment service.
 *
 * <p>Note that this preference is not persisted by the OS, and hence must be
 * called every time the Activity is resumed.
 *
 * @param activity The activity which prefers this service to be invoked
 * @param service The service to be preferred while this activity is in the foreground
 * @return whether the registration was successful
 */

public boolean setPreferredService(android.app.Activity activity, android.content.ComponentName service) { throw new RuntimeException("Stub!"); }

/**
 * Sets whether when this service becomes the preferred service, if the NFC stack
 * should enable observe mode or disable observe mode. The default is to not enable observe
 * mode when a service either the foreground default service or the default payment service so
 * not calling this method will preserve that behavior.
 *
 * @param service The component name of the service
 * This value cannot be {@code null}.
 * @param enable Whether the service should default to observe mode or not
 * @return whether the change was successful.
 */

public boolean setShouldDefaultToObserveModeForService(@android.annotation.NonNull android.content.ComponentName service, boolean enable) { throw new RuntimeException("Stub!"); }

/**
 * Some devices may allow an application to register all
 * AIDs that starts with a certain prefix, e.g.
 * "A000000004*" to register all MasterCard AIDs.
 *
 * Use this method to determine whether this device
 * supports registering AID prefixes.
 *
 * @return whether AID prefix registering is supported on this device.
 */

public boolean supportsAidPrefixRegistration() { throw new RuntimeException("Stub!"); }

/**
 * Unregister a preferred service listener that was previously registered with {@link
 * #registerNfcEventCallback(java.util.concurrent.Executor,android.nfc.cardemulation.CardEmulation.NfcEventCallback)}
 *
 * @param listener The previously registered listener to unregister

 * This value cannot be {@code null}.
 */

public void unregisterNfcEventCallback(@android.annotation.NonNull android.nfc.cardemulation.CardEmulation.NfcEventCallback listener) { throw new RuntimeException("Stub!"); }

/**
 * Unsets the off-host Secure Element for the given service.
 *
 * <p>Note that this will only remove Secure Element that was dynamically
 * set using the {@link #setOffHostForService(android.content.ComponentName,java.lang.String)}
 * and resets it to a value that was statically assigned using manifest.
 *
 * <p>Note that you can only unset off-host SE for a service that
 * is running under the same UID as the caller of this API. Typically
 * this means you need to call this from the same
 * package as the service itself, though UIDs can also
 * be shared between packages using shared UIDs.
 *
 * <br>
 * Requires {@link android.Manifest.permission#NFC}
 * @param service The component name of the service
 * This value cannot be {@code null}.
 * @return whether the registration was successful.
 
 * This value cannot be {@code null}.
 */

@android.annotation.NonNull
public boolean unsetOffHostForService(@android.annotation.NonNull android.content.ComponentName service) { throw new RuntimeException("Stub!"); }

/**
 * Unsets the preferred service for the specified Activity.
 *
 * <p>Note that the specified Activity must still be in resumed
 * state at the time of this call. A good place to call this method
 * is in your {@link android.app.Activity#onPause Activity.onPause} implementation.
 *
 * @param activity The activity which the service was registered for
 * @return true when successful
 */

public boolean unsetPreferredService(android.app.Activity activity) { throw new RuntimeException("Stub!"); }

/**
 * Activity action: ask the user to change the default
 * card emulation service for a certain category. This will
 * show a dialog that asks the user whether they want to
 * replace the current default service with the service
 * identified with the ComponentName specified in
 * {@link #EXTRA_SERVICE_COMPONENT}, for the category
 * specified in {@link #EXTRA_CATEGORY}. There is an optional
 * extra field using {@link android.content.Intent#EXTRA_USER Intent.EXTRA_USER} to specify
 * the {@link android.os.UserHandle UserHandle} of the user that owns the app.
 *
 * @deprecated Please use {@link android.app.role.RoleManager#createRequestRoleIntent(String)}
 * with {@link android.app.role.RoleManager#ROLE_WALLET} parameter
 * and {@link android.app.Activity#startActivityForResult(android.content.Intent,int) Activity.startActivityForResult(Intent, int)} instead.
 */

@Deprecated @android.annotation.SdkConstant(android.annotation.SdkConstant.SdkConstantType.ACTIVITY_INTENT_ACTION) public static final java.lang.String ACTION_CHANGE_DEFAULT = "android.nfc.cardemulation.action.ACTION_CHANGE_DEFAULT";

/**
 * Category that can be used for all other card emulation
 * services.
 */

public static final java.lang.String CATEGORY_OTHER = "other";

/**
 * Category used for NFC payment services.
 */

public static final java.lang.String CATEGORY_PAYMENT = "payment";

/**
 * The category extra for {@link #ACTION_CHANGE_DEFAULT}.
 *
 * @see #ACTION_CHANGE_DEFAULT
 */

public static final java.lang.String EXTRA_CATEGORY = "category";

/**
 * The service {@link android.content.ComponentName ComponentName} object passed in as an
 * extra for {@link #ACTION_CHANGE_DEFAULT}.
 *
 * @see #ACTION_CHANGE_DEFAULT
 */

public static final java.lang.String EXTRA_SERVICE_COMPONENT = "component";

/**
 * This error is reported when the NFC stack times out while waiting for a response to a command
 * sent to the NFC hardware.
 */

public static final int NFC_INTERNAL_ERROR_COMMAND_TIMEOUT = 3; // 0x3

/**
 * This error is reported when the NFC command watchdog restarts the NFC stack.
 */

public static final int NFC_INTERNAL_ERROR_NFC_CRASH_RESTART = 1; // 0x1

/**
 * This error is reported when the NFC controller does not respond or there's an NCI transport
 * error.
 */

public static final int NFC_INTERNAL_ERROR_NFC_HARDWARE_ERROR = 2; // 0x2

public static final int NFC_INTERNAL_ERROR_UNKNOWN = 0; // 0x0

/**
 * Property name used to indicate that an application wants to allow associated services
 * to share the same AID routing priority when this application is the role holder.
 * <p>
 * Example:
 * <pre>
 *     {@code
 *     <application>
 *       ...
 *       <property android:name="android.nfc.cardemulation.PROPERTY_ALLOW_SHARED_ROLE_PRIORITY"
 *         android:value="true"/>
 *     </application>
 *     }
 * </pre>
 */

public static final java.lang.String PROPERTY_ALLOW_SHARED_ROLE_PRIORITY = "android.nfc.cardemulation.PROPERTY_ALLOW_SHARED_ROLE_PRIORITY";

/**
 * Route to the default value in config file.
 */

public static final int PROTOCOL_AND_TECHNOLOGY_ROUTE_DEFAULT = 3; // 0x3

/**
 * Route to Device Host (DH).
 */

public static final int PROTOCOL_AND_TECHNOLOGY_ROUTE_DH = 0; // 0x0

/**
 * Route to eSE.
 */

public static final int PROTOCOL_AND_TECHNOLOGY_ROUTE_ESE = 1; // 0x1

/**
 * Route to UICC.
 */

public static final int PROTOCOL_AND_TECHNOLOGY_ROUTE_UICC = 2; // 0x2

/**
 * Route unset.
 */

public static final int PROTOCOL_AND_TECHNOLOGY_ROUTE_UNSET = -1; // 0xffffffff

/**
 * Return value for {@link #getSelectionModeForCategory(java.lang.String)}.
 *
 * <p>In this mode, when using ISO-DEP card emulation with {@link android.nfc.cardemulation.HostApduService HostApduService}
 *    or {@link android.nfc.cardemulation.OffHostApduService OffHostApduService}, whenever an Application ID (AID) of this category
 *    is selected, the user is asked which service they want to use to handle
 *    the transaction, even if there is only one matching service.
 */

public static final int SELECTION_MODE_ALWAYS_ASK = 1; // 0x1

/**
 * Return value for {@link #getSelectionModeForCategory(java.lang.String)}.
 *
 * <p>In this mode, when using ISO-DEP card emulation with {@link android.nfc.cardemulation.HostApduService HostApduService}
 *    or {@link android.nfc.cardemulation.OffHostApduService OffHostApduService}, the user will only be asked to select a service
 *    if the Application ID (AID) selected by the reader has been registered by multiple
 *    services. If there is only one service that has registered for the AID,
 *    that service will be invoked directly.
 */

public static final int SELECTION_MODE_ASK_IF_CONFLICT = 2; // 0x2

/**
 * Return value for {@link #getSelectionModeForCategory(java.lang.String)}.
 *
 * <p>In this mode, the user has set a default service for this
 *    category.
 *
 * <p>When using ISO-DEP card emulation with {@link android.nfc.cardemulation.HostApduService HostApduService}
 *    or {@link android.nfc.cardemulation.OffHostApduService OffHostApduService}, if a remote NFC device selects
 *    any of the Application IDs (AIDs)
 *    that the default service has registered in this category,
 *    that service will automatically be bound to to handle
 *    the transaction.
 */

public static final int SELECTION_MODE_PREFER_DEFAULT = 0; // 0x0
/** Listener for preferred service state changes. */

@SuppressWarnings({"unchecked", "deprecation", "all"})
public static interface NfcEventCallback {

/**
 * This method is called when an AID conflict is detected during an NFC transaction. This
 * can happen when multiple services are registered for the same AID. If your service is
 * registered for this AID you may want to instruct users to bring your app to the
 * foreground and ensure you call {@link #setPreferredService(android.app.Activity,android.content.ComponentName)}
 * to ensure the transaction is routed to your service.
 *
 * @param aid The AID that is in conflict

 * This value cannot be {@code null}.
 */

public default void onAidConflictOccurred(@android.annotation.NonNull java.lang.String aid) { throw new RuntimeException("Stub!"); }

/**
 * This method is called when an AID is not routed to any service during an NFC
 * transaction. This can happen when no service is registered for the given AID.
 *
 * @param aid the AID that was not routed

 * This value cannot be {@code null}.
 */

public default void onAidNotRouted(@android.annotation.NonNull java.lang.String aid) { throw new RuntimeException("Stub!"); }

/**
 * This method is called when an internal error is reported by the NFC stack.
 *
 * No action is required in response to these events as the NFC stack will automatically
 * attempt to recover. These errors are reported for informational purposes only.
 *
 * Note that these errors can be reported when performing various internal NFC operations
 * (such as during device shutdown) and cannot always be explicitly correlated with NFC
 * transaction failures.
 *
 * @param errorType The type of the internal error

 * Value is {@link android.nfc.cardemulation.CardEmulation#NFC_INTERNAL_ERROR_UNKNOWN}, {@link android.nfc.cardemulation.CardEmulation#NFC_INTERNAL_ERROR_NFC_CRASH_RESTART}, {@link android.nfc.cardemulation.CardEmulation#NFC_INTERNAL_ERROR_NFC_HARDWARE_ERROR}, or {@link android.nfc.cardemulation.CardEmulation#NFC_INTERNAL_ERROR_COMMAND_TIMEOUT}
 */

public default void onInternalErrorReported(int errorType) { throw new RuntimeException("Stub!"); }

/**
 * This method is called when the NFC state changes.
 *
 * @see android.nfc.NfcAdapter#getAdapterState()
 *
 * @param state The new NFC state

 * Value is {@link android.nfc.NfcAdapter#STATE_OFF}, {@link android.nfc.NfcAdapter#STATE_TURNING_ON}, {@link android.nfc.NfcAdapter#STATE_ON}, or {@link android.nfc.NfcAdapter#STATE_TURNING_OFF}
 */

public default void onNfcStateChanged(int state) { throw new RuntimeException("Stub!"); }

/**
 * This method is called when observe mode has been enabled or disabled.
 *
 * @param isEnabled true if observe mode has been enabled, false if it has been disabled
 */

public default void onObserveModeStateChanged(boolean isEnabled) { throw new RuntimeException("Stub!"); }

/**
 * This method is called when this package gains or loses preferred Nfc service status,
 * either the Default Wallet Role holder (see {@link
 * android.app.role.RoleManager#ROLE_WALLET}) or the preferred service of the foreground
 * activity set with {@link #setPreferredService(android.app.Activity,android.content.ComponentName)}
 *
 * @param isPreferred true is this service has become the preferred Nfc service, false if it
 *     is no longer the preferred service
 */

public default void onPreferredServiceChanged(boolean isPreferred) { throw new RuntimeException("Stub!"); }

/**
 * This method is called when the NFC controller is in card emulation mode and an NFC
 * reader's field is either detected or lost.
 *
 * @param isDetected true if an NFC reader is detected, false if it is lost
 */

public default void onRemoteFieldChanged(boolean isDetected) { throw new RuntimeException("Stub!"); }
}

}

