/*
 * Copyright (C) 2011 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */


package android.net.wifi.p2p;

import android.content.Context;
import android.net.MacAddress;
import android.net.NetworkInfo;
import android.net.wifi.WpsInfo;
import android.net.wifi.p2p.nsd.WifiP2pDnsSdServiceInfo;
import android.net.wifi.p2p.nsd.WifiP2pServiceInfo;
import android.net.wifi.p2p.nsd.WifiP2pServiceRequest;
import android.net.wifi.p2p.nsd.WifiP2pUpnpServiceInfo;
import android.net.wifi.p2p.nsd.WifiP2pUsdBasedServiceConfig;
import android.net.wifi.p2p.nsd.WifiP2pUsdBasedServiceResponse;
import android.os.Build;
import android.os.Looper;
import android.os.OutcomeReceiver;
import android.view.Display;
import java.util.List;
import java.util.concurrent.Executor;
import java.util.function.Consumer;

/**
 * This class provides the API for managing Wi-Fi peer-to-peer connectivity. This lets an
 * application discover available peers, setup connection to peers and query for the list of peers.
 * When a p2p connection is formed over wifi, the device continues to maintain the uplink
 * connection over mobile or any other available network for internet connectivity on the device.
 *
 * <p> The API is asynchronous and responses to requests from an application are on listener
 * callbacks provided by the application. The application needs to do an initialization with
 * {@link #initialize} before doing any p2p operation.
 *
 * <p> Most application calls need a {@link android.net.wifi.p2p.WifiP2pManager.ActionListener ActionListener} instance for receiving callbacks
 * {@link android.net.wifi.p2p.WifiP2pManager.ActionListener#onSuccess ActionListener.onSuccess} or {@link android.net.wifi.p2p.WifiP2pManager.ActionListener#onFailure ActionListener.onFailure}. Action callbacks
 * indicate whether the initiation of the action was a success or a failure.
 * Upon failure, the reason of failure can be one of {@link #ERROR}, {@link #P2P_UNSUPPORTED}
 * or {@link #BUSY}.
 *
 * <p> An application can initiate discovery of peers with {@link #discoverPeers}. An initiated
 * discovery request from an application stays active until the device starts connecting to a peer
 * ,forms a p2p group or there is an explicit {@link #stopPeerDiscovery}.
 * Applications can listen to {@link #WIFI_P2P_DISCOVERY_CHANGED_ACTION} to know if a peer-to-peer
 * discovery is running or stopped. Additionally, {@link #WIFI_P2P_PEERS_CHANGED_ACTION} indicates
 * if the peer list has changed.
 *
 * <p> When an application needs to fetch the current list of peers, it can request the list
 * of peers with {@link #requestPeers}. When the peer list is available
 * {@link android.net.wifi.p2p.WifiP2pManager.PeerListListener#onPeersAvailable PeerListListener.onPeersAvailable} is called with the device list.
 *
 * <p> An application can initiate a connection request to a peer through {@link #connect}. See
 * {@link android.net.wifi.p2p.WifiP2pConfig WifiP2pConfig} for details on setting up the configuration. For communication with legacy
 * Wi-Fi devices that do not support p2p, an app can create a group using {@link #createGroup}
 * which creates an access point whose details can be fetched with {@link #requestGroupInfo}.
 *
 * <p> After a successful group formation through {@link #createGroup} or through {@link #connect},
 * use {@link #requestConnectionInfo} to fetch the connection details. The connection info
 * {@link android.net.wifi.p2p.WifiP2pInfo WifiP2pInfo} contains the address of the group owner
 * {@link android.net.wifi.p2p.WifiP2pInfo#groupOwnerAddress WifiP2pInfo.groupOwnerAddress} and a flag {@link android.net.wifi.p2p.WifiP2pInfo#isGroupOwner WifiP2pInfo.isGroupOwner} to indicate
 * if the current device is a p2p group owner. A p2p client can thus communicate with
 * the p2p group owner through a socket connection. If the current device is the p2p group owner,
 * {@link android.net.wifi.p2p.WifiP2pInfo#groupOwnerAddress WifiP2pInfo.groupOwnerAddress} is anonymized unless the caller holds the
 * {@code android.Manifest.permission#LOCAL_MAC_ADDRESS} permission.
 *
 * <p> With peer discovery using {@link  #discoverPeers}, an application discovers the neighboring
 * peers, but has no good way to figure out which peer to establish a connection with. For example,
 * if a game application is interested in finding all the neighboring peers that are also running
 * the same game, it has no way to find out until after the connection is setup. Pre-association
 * service discovery is meant to address this issue of filtering the peers based on the running
 * services.
 *
 * <p>With pre-association service discovery, an application can advertise a service for a
 * application on a peer device prior to a connection setup between the devices.
 * Currently, DNS based service discovery (Bonjour) and Upnp are the higher layer protocols
 * supported. Get Bonjour resources at dns-sd.org and Upnp resources at upnp.org
 * As an example, a video application can discover a Upnp capable media renderer
 * prior to setting up a Wi-fi p2p connection with the device.
 *
 * <p> An application can advertise a Upnp or a Bonjour service with a call to
 * {@link #addLocalService}. After a local service is added,
 * the framework automatically responds to a peer application discovering the service prior
 * to establishing a p2p connection. A call to {@link #removeLocalService} removes a local
 * service and {@link #clearLocalServices} can be used to clear all local services.
 *
 * <p> An application that is looking for peer devices that support certain services
 * can do so with a call to  {@link #discoverServices}. Prior to initiating the discovery,
 * application can add service discovery request with a call to {@link #addServiceRequest},
 * remove a service discovery request with a call to {@link #removeServiceRequest} or clear
 * all requests with a call to {@link #clearServiceRequests}. When no service requests remain,
 * a previously running service discovery will stop.
 *
 * The application is notified of a result of service discovery request through listener callbacks
 * set through {@link #setDnsSdResponseListeners} for Bonjour or
 * {@link #setUpnpServiceResponseListener} for Upnp.
 *
 * <p class="note"><strong>Note:</strong>
 * Registering an application handler with {@link #initialize} requires the permissions
 * {@link android.Manifest.permission#ACCESS_WIFI_STATE} and
 * {@link android.Manifest.permission#CHANGE_WIFI_STATE} to perform any further peer-to-peer
 * operations.
 *
 * {@see WifiP2pConfig}
 * {@see WifiP2pInfo}
 * {@see WifiP2pGroup}
 * {@see WifiP2pDevice}
 * {@see WifiP2pDeviceList}
 * {@see android.net.wifi.WpsInfo}
 */

@SuppressWarnings({"unchecked", "deprecation", "all"})
public class WifiP2pManager {

WifiP2pManager() { throw new RuntimeException("Stub!"); }

/**
 * Set the external approver for a specific peer.
 *
 * This API associates a specific peer with an approver. When an incoming request is received
 * from a peer, an authorization request is routed to the attached approver. The approver then
 * calls {@link #setConnectionRequestResult(android.net.wifi.p2p.WifiP2pManager.Channel,android.net.MacAddress,int,android.net.wifi.p2p.WifiP2pManager.ActionListener)} to send
 * the result to the WiFi service. A specific peer (identified by its {@code MacAddress}) can
 * only be attached to a single approver. The previous approver will be detached once a new
 * approver is attached. The approver will also be detached automatically when the channel is
 * closed.
 * <p>
 * When an approver is attached, {@link android.net.wifi.p2p.WifiP2pManager.ExternalApproverRequestListener#onAttached(android.net.MacAddress) ExternalApproverRequestListener.onAttached(MacAddress)}
 * is called. When an approver is detached,
 * {@link android.net.wifi.p2p.WifiP2pManager.ExternalApproverRequestListener#onDetached(android.net.MacAddress,int) ExternalApproverRequestListener.onDetached(MacAddress, int)} is called.
 * When an incoming request is received,
 * {@link android.net.wifi.p2p.WifiP2pManager.ExternalApproverRequestListener#onConnectionRequested(int,android.net.wifi.p2p.WifiP2pConfig,android.net.wifi.p2p.WifiP2pDevice) ExternalApproverRequestListener.onConnectionRequested(int, WifiP2pConfig, WifiP2pDevice)}
 * is called. When a WPS PIN is generated,
 * {@link android.net.wifi.p2p.WifiP2pManager.ExternalApproverRequestListener#onPinGenerated(android.net.MacAddress,java.lang.String) ExternalApproverRequestListener.onPinGenerated(MacAddress, String)} is called.
 * <p>
 * The application must have {@link android.Manifest.permission#NEARBY_WIFI_DEVICES} with
 * android:usesPermissionFlags="neverForLocation". If the application does not declare
 * android:usesPermissionFlags="neverForLocation", then it must also have
 * {@link android.Manifest.permission#ACCESS_FINE_LOCATION}.
 *
 * <br>
 * Requires {@link android.Manifest.permission#MANAGE_WIFI_NETWORK_SELECTION}
 * @param c is the channel created at {@link #initialize(android.content.Context,android.os.Looper,android.net.wifi.p2p.WifiP2pManager.ChannelListener)}.
 * This value cannot be {@code null}.
 * @param deviceAddress the peer which is bound to the external approver.
 * This value cannot be {@code null}.
 * @param listener for callback when the framework needs to notify the external approver.

 * This value cannot be {@code null}.
 */

public void addExternalApprover(@android.annotation.NonNull android.net.wifi.p2p.WifiP2pManager.Channel c, @android.annotation.NonNull android.net.MacAddress deviceAddress, @android.annotation.NonNull android.net.wifi.p2p.WifiP2pManager.ExternalApproverRequestListener listener) { throw new RuntimeException("Stub!"); }

/**
 * Register a local service for service discovery. If a local service is registered,
 * the framework automatically responds to a service discovery request from a peer.
 *
 * <p> The function call immediately returns after sending a request to add a local
 * service to the framework. The application is notified of a success or failure to
 * add service through listener callbacks {@link android.net.wifi.p2p.WifiP2pManager.ActionListener#onSuccess ActionListener.onSuccess} or
 * {@link android.net.wifi.p2p.WifiP2pManager.ActionListener#onFailure ActionListener.onFailure}.
 *
 * <p>The service information is set through {@link android.net.wifi.p2p.nsd.WifiP2pServiceInfo WifiP2pServiceInfo}.<br>
 * or its subclass calls  {@link android.net.wifi.p2p.nsd.WifiP2pUpnpServiceInfo#newInstance WifiP2pUpnpServiceInfo.newInstance} or
 *  {@link android.net.wifi.p2p.nsd.WifiP2pDnsSdServiceInfo#newInstance WifiP2pDnsSdServiceInfo.newInstance} for a Upnp or Bonjour service
 * respectively
 *
 * <p>The service information can be cleared with calls to
 *  {@link #removeLocalService} or {@link #clearLocalServices}.
 * <p>
 * If targeting {@link android.os.Build.VERSION_CODES#TIRAMISU} or later, the application must
 * have {@link android.Manifest.permission#NEARBY_WIFI_DEVICES} with
 * android:usesPermissionFlags="neverForLocation". If the application does not declare
 * android:usesPermissionFlags="neverForLocation", then it must also have
 * {@link android.Manifest.permission#ACCESS_FINE_LOCATION}.
 *
 * If targeting an earlier release than {@link android.os.Build.VERSION_CODES#TIRAMISU}, the
 * application must have {@link android.Manifest.permission#ACCESS_FINE_LOCATION}.
 *
 * @param channel is the channel created at {@link #initialize}
 * @param servInfo is a local service information.
 * @param listener for callbacks on success or failure. Can be null.
 */

public void addLocalService(android.net.wifi.p2p.WifiP2pManager.Channel channel, android.net.wifi.p2p.nsd.WifiP2pServiceInfo servInfo, android.net.wifi.p2p.WifiP2pManager.ActionListener listener) { throw new RuntimeException("Stub!"); }

/**
 * Add a service discovery request.
 *
 * <p> The function call immediately returns after sending a request to add service
 * discovery request to the framework. The application is notified of a success or failure to
 * add service through listener callbacks {@link android.net.wifi.p2p.WifiP2pManager.ActionListener#onSuccess ActionListener.onSuccess} or
 * {@link android.net.wifi.p2p.WifiP2pManager.ActionListener#onFailure ActionListener.onFailure}.
 *
 * <p> The USD based service information are set in the service request through
 * {@link android.net.wifi.p2p.nsd.WifiP2pServiceRequest#WifiP2pServiceRequest(android.net.wifi.p2p.nsd.WifiP2pUsdBasedServiceConfig) WifiP2pServiceRequest.WifiP2pServiceRequest(WifiP2pUsdBasedServiceConfig)}.
 * Application must use {@link #isWiFiDirectR2Supported()} to determine whether the device
 * supports USD based service discovery. If {@link #isWiFiDirectR2Supported()} return
 * {@code false} then this method will throw {@link java.lang.UnsupportedOperationException UnsupportedOperationException} for service
 * request information containing USD service configuration.
 *
 * <p>After service discovery request is added, you can initiate service discovery by
 * {@link #discoverServices}.
 *
 * <p>The added service requests can be cleared with calls to
 * {@link #removeServiceRequest(android.net.wifi.p2p.WifiP2pManager.Channel,android.net.wifi.p2p.nsd.WifiP2pServiceRequest,android.net.wifi.p2p.WifiP2pManager.ActionListener)} or
 * {@link #clearServiceRequests(android.net.wifi.p2p.WifiP2pManager.Channel,android.net.wifi.p2p.WifiP2pManager.ActionListener)}.
 *
 * @param channel is the channel created at {@link #initialize}
 * @param req is the service discovery request.
 * @param listener for callbacks on success or failure. Can be null.
 */

public void addServiceRequest(android.net.wifi.p2p.WifiP2pManager.Channel channel, android.net.wifi.p2p.nsd.WifiP2pServiceRequest req, android.net.wifi.p2p.WifiP2pManager.ActionListener listener) { throw new RuntimeException("Stub!"); }

/**
 * Cancel any ongoing p2p group negotiation
 *
 * <p> The function call immediately returns after sending a connection cancellation request
 * to the framework. The application is notified of a success or failure to initiate
 * cancellation through listener callbacks {@link android.net.wifi.p2p.WifiP2pManager.ActionListener#onSuccess ActionListener.onSuccess} or
 * {@link android.net.wifi.p2p.WifiP2pManager.ActionListener#onFailure ActionListener.onFailure}.
 *
 * @param channel is the channel created at {@link #initialize}
 * @param listener for callbacks on success or failure. Can be null.
 */

public void cancelConnect(android.net.wifi.p2p.WifiP2pManager.Channel channel, android.net.wifi.p2p.WifiP2pManager.ActionListener listener) { throw new RuntimeException("Stub!"); }

/**
 * Clear all registered local services of service discovery.
 *
 * <p> The function call immediately returns after sending a request to clear all
 * local services to the framework. The application is notified of a success or failure to
 * add service through listener callbacks {@link android.net.wifi.p2p.WifiP2pManager.ActionListener#onSuccess ActionListener.onSuccess} or
 * {@link android.net.wifi.p2p.WifiP2pManager.ActionListener#onFailure ActionListener.onFailure}.
 *
 * @param channel is the channel created at {@link #initialize}
 * @param listener for callbacks on success or failure. Can be null.
 */

public void clearLocalServices(android.net.wifi.p2p.WifiP2pManager.Channel channel, android.net.wifi.p2p.WifiP2pManager.ActionListener listener) { throw new RuntimeException("Stub!"); }

/**
 * Clear all registered service discovery requests.
 *
 * <p> The function call immediately returns after sending a request to clear all
 * service discovery requests to the framework. The application is notified of a success
 * or failure to add service through listener callbacks {@link android.net.wifi.p2p.WifiP2pManager.ActionListener#onSuccess ActionListener.onSuccess} or
 * {@link android.net.wifi.p2p.WifiP2pManager.ActionListener#onFailure ActionListener.onFailure}.
 *
 * @param channel is the channel created at {@link #initialize}
 * @param listener for callbacks on success or failure. Can be null.
 */

public void clearServiceRequests(android.net.wifi.p2p.WifiP2pManager.Channel channel, android.net.wifi.p2p.WifiP2pManager.ActionListener listener) { throw new RuntimeException("Stub!"); }

/**
 * Start a p2p connection to a device with the specified configuration.
 *
 * <p> The function call immediately returns after sending a connection request
 * to the framework. The application is notified of a success or failure to initiate
 * connect through listener callbacks {@link android.net.wifi.p2p.WifiP2pManager.ActionListener#onSuccess ActionListener.onSuccess} or
 * {@link android.net.wifi.p2p.WifiP2pManager.ActionListener#onFailure ActionListener.onFailure}.
 *
 * <p> An app should use {@link android.net.wifi.p2p.WifiP2pConfig.Builder WifiP2pConfig.Builder} to build the configuration
 * for this API, ex. call {@link android.net.wifi.p2p.WifiP2pConfig.Builder#setDeviceAddress(android.net.MacAddress) WifiP2pConfig.Builder.setDeviceAddress(MacAddress)}
 * to set the peer MAC address and {@link android.net.wifi.p2p.WifiP2pConfig.Builder#enablePersistentMode(boolean) WifiP2pConfig.Builder.enablePersistentMode(boolean)}
 * to configure the persistent mode.
 *
 * <p> Register for {@link #WIFI_P2P_CONNECTION_CHANGED_ACTION} intent to
 * determine when the framework notifies of a change in connectivity.
 *
 * <p> If the current device is not part of a p2p group, a connect request initiates
 * a group negotiation with the peer.
 *
 * <p> If the current device is part of an existing p2p group or has created
 * a p2p group with {@link #createGroup}, an invitation to join the group is sent to
 * the peer device.
 * <p>
 * If targeting {@link android.os.Build.VERSION_CODES#TIRAMISU} or later, the application must
 * have {@link android.Manifest.permission#NEARBY_WIFI_DEVICES} with
 * android:usesPermissionFlags="neverForLocation". If the application does not declare
 * android:usesPermissionFlags="neverForLocation", then it must also have
 * {@link android.Manifest.permission#ACCESS_FINE_LOCATION}.
 *
 * If targeting an earlier release than {@link android.os.Build.VERSION_CODES#TIRAMISU}, the
 * application must have {@link android.Manifest.permission#ACCESS_FINE_LOCATION}.
 *
 * @param channel is the channel created at {@link #initialize}
 * @param config options as described in {@link android.net.wifi.p2p.WifiP2pConfig WifiP2pConfig} class
 * @param listener for callbacks on success or failure. Can be null.
 */

public void connect(android.net.wifi.p2p.WifiP2pManager.Channel channel, android.net.wifi.p2p.WifiP2pConfig config, android.net.wifi.p2p.WifiP2pManager.ActionListener listener) { throw new RuntimeException("Stub!"); }

/**
 * Create a p2p group with the current device as the group owner. This essentially creates
 * an access point that can accept connections from legacy clients as well as other p2p
 * devices.
 *
 * <p> An app should use {@link android.net.wifi.p2p.WifiP2pConfig.Builder WifiP2pConfig.Builder} to build the configuration
 * for a group.
 *
 * <p class="note"><strong>Note:</strong>
 * This function would normally not be used unless the current device needs
 * to form a p2p group as a Group Owner and allow peers to join it as either
 * Group Clients or legacy Wi-Fi STAs.
 *
 * <p> The function call immediately returns after sending a group creation request
 * to the framework. The application is notified of a success or failure to initiate
 * group creation through listener callbacks {@link android.net.wifi.p2p.WifiP2pManager.ActionListener#onSuccess ActionListener.onSuccess} or
 * {@link android.net.wifi.p2p.WifiP2pManager.ActionListener#onFailure ActionListener.onFailure}.
 *
 * <p> Application can request for the group details with {@link #requestGroupInfo}.
 * <p>
 * If targeting {@link android.os.Build.VERSION_CODES#TIRAMISU} or later, the application must
 * have {@link android.Manifest.permission#NEARBY_WIFI_DEVICES} with
 * android:usesPermissionFlags="neverForLocation". If the application does not declare
 * android:usesPermissionFlags="neverForLocation", then it must also have
 * {@link android.Manifest.permission#ACCESS_FINE_LOCATION}.
 *
 * If targeting an earlier release than {@link android.os.Build.VERSION_CODES#TIRAMISU}, the
 * application must have {@link android.Manifest.permission#ACCESS_FINE_LOCATION}.
 *
 * @param channel is the channel created at {@link #initialize}.
 * This value cannot be {@code null}.
 * @param config the configuration of a p2p group.
 * This value may be {@code null}.
 * @param listener for callbacks on success or failure. Can be null.
 */

public void createGroup(@android.annotation.NonNull android.net.wifi.p2p.WifiP2pManager.Channel channel, @android.annotation.Nullable android.net.wifi.p2p.WifiP2pConfig config, @android.annotation.Nullable android.net.wifi.p2p.WifiP2pManager.ActionListener listener) { throw new RuntimeException("Stub!"); }

/**
 * Create a p2p group with the current device as the group owner. This essentially creates
 * an access point that can accept connections from legacy clients as well as other p2p
 * devices.
 *
 * <p class="note"><strong>Note:</strong>
 * This function would normally not be used unless the current device needs
 * to form a p2p connection with a legacy client
 *
 * <p> The function call immediately returns after sending a group creation request
 * to the framework. The application is notified of a success or failure to initiate
 * group creation through listener callbacks {@link android.net.wifi.p2p.WifiP2pManager.ActionListener#onSuccess ActionListener.onSuccess} or
 * {@link android.net.wifi.p2p.WifiP2pManager.ActionListener#onFailure ActionListener.onFailure}.
 *
 * <p> Application can request for the group details with {@link #requestGroupInfo}.
 * <p>
 * If targeting {@link android.os.Build.VERSION_CODES#TIRAMISU} or later, the application must
 * have {@link android.Manifest.permission#NEARBY_WIFI_DEVICES} with
 * android:usesPermissionFlags="neverForLocation". If the application does not declare
 * android:usesPermissionFlags="neverForLocation", then it must also have
 * {@link android.Manifest.permission#ACCESS_FINE_LOCATION}.
 *
 * If targeting an earlier release than {@link android.os.Build.VERSION_CODES#TIRAMISU}, the
 * application must have {@link android.Manifest.permission#ACCESS_FINE_LOCATION}.
 *
 * @param channel is the channel created at {@link #initialize}
 * @param listener for callbacks on success or failure. Can be null.
 */

public void createGroup(android.net.wifi.p2p.WifiP2pManager.Channel channel, android.net.wifi.p2p.WifiP2pManager.ActionListener listener) { throw new RuntimeException("Stub!"); }

/**
 * Initiate peer discovery. A discovery process involves scanning for available Wi-Fi peers
 * for the purpose of establishing a connection.
 *
 * <p> The function call immediately returns after sending a discovery request
 * to the framework. The application is notified of a success or failure to initiate
 * discovery through listener callbacks {@link android.net.wifi.p2p.WifiP2pManager.ActionListener#onSuccess ActionListener.onSuccess} or
 * {@link android.net.wifi.p2p.WifiP2pManager.ActionListener#onFailure ActionListener.onFailure}.
 *
 * <p> The discovery remains active until a connection is initiated or
 * a p2p group is formed. Register for {@link #WIFI_P2P_PEERS_CHANGED_ACTION} intent to
 * determine when the framework notifies of a change as peers are discovered.
 *
 * <p> Upon receiving a {@link #WIFI_P2P_PEERS_CHANGED_ACTION} intent, an application
 * can request the list of peers using {@link #requestPeers}.
 * <p>
 * If targeting {@link android.os.Build.VERSION_CODES#TIRAMISU} or later, the application must
 * have {@link android.Manifest.permission#NEARBY_WIFI_DEVICES} with
 * android:usesPermissionFlags="neverForLocation". If the application does not declare
 * android:usesPermissionFlags="neverForLocation", then it must also have
 * {@link android.Manifest.permission#ACCESS_FINE_LOCATION}.
 *
 * If targeting an earlier release than {@link android.os.Build.VERSION_CODES#TIRAMISU}, the
 * application must have {@link android.Manifest.permission#ACCESS_FINE_LOCATION}.
 *
 * @param channel is the channel created at {@link #initialize}
 * @param listener for callbacks on success or failure. Can be null.
 */

public void discoverPeers(android.net.wifi.p2p.WifiP2pManager.Channel channel, android.net.wifi.p2p.WifiP2pManager.ActionListener listener) { throw new RuntimeException("Stub!"); }

/**
 * Scan only the social channels.
 *
 * A discovery process involves scanning for available Wi-Fi peers
 * for the purpose of establishing a connection.
 *
 * <p> The function call immediately returns after sending a discovery request
 * to the framework. The application is notified of a success or failure to initiate
 * discovery through listener callbacks {@link android.net.wifi.p2p.WifiP2pManager.ActionListener#onSuccess ActionListener.onSuccess} or
 * {@link android.net.wifi.p2p.WifiP2pManager.ActionListener#onFailure ActionListener.onFailure}.
 *
 * <p> The discovery remains active until a connection is initiated or
 * a p2p group is formed. Register for {@link #WIFI_P2P_PEERS_CHANGED_ACTION} intent to
 * determine when the framework notifies of a change as peers are discovered.
 *
 * <p> Upon receiving a {@link #WIFI_P2P_PEERS_CHANGED_ACTION} intent, an application
 * can request the list of peers using {@link #requestPeers}.
 * <p>
 * The application must have {@link android.Manifest.permission#NEARBY_WIFI_DEVICES} with
 * android:usesPermissionFlags="neverForLocation". If the application does not declare
 * android:usesPermissionFlags="neverForLocation", then it must also have
 * {@link android.Manifest.permission#ACCESS_FINE_LOCATION}.
 * <p>
 * Use {@link #isChannelConstrainedDiscoverySupported()} to determine whether the device
 * supports this feature. If {@link #isChannelConstrainedDiscoverySupported()} return
 * {@code false} then this method will throw {@link java.lang.UnsupportedOperationException UnsupportedOperationException}.
 *
 * @param channel is the channel created at {@link #initialize}
 * This value cannot be {@code null}.
 * @param listener for callbacks on success or failure.

 * This value may be {@code null}.
 */

public void discoverPeersOnSocialChannels(@android.annotation.NonNull android.net.wifi.p2p.WifiP2pManager.Channel channel, @android.annotation.Nullable android.net.wifi.p2p.WifiP2pManager.ActionListener listener) { throw new RuntimeException("Stub!"); }

/**
 * Scan only a single channel specified by frequency.
 *
 * A discovery process involves scanning for available Wi-Fi peers
 * for the purpose of establishing a connection.
 *
 * <p> The function call immediately returns after sending a discovery request
 * to the framework. The application is notified of a success or failure to initiate
 * discovery through listener callbacks {@link android.net.wifi.p2p.WifiP2pManager.ActionListener#onSuccess ActionListener.onSuccess} or
 * {@link android.net.wifi.p2p.WifiP2pManager.ActionListener#onFailure ActionListener.onFailure}.
 *
 * <p> The discovery remains active until a connection is initiated or
 * a p2p group is formed. Register for {@link #WIFI_P2P_PEERS_CHANGED_ACTION} intent to
 * determine when the framework notifies of a change as peers are discovered.
 *
 * <p> Upon receiving a {@link #WIFI_P2P_PEERS_CHANGED_ACTION} intent, an application
 * can request the list of peers using {@link #requestPeers}.
 * <p>
 * The application must have {@link android.Manifest.permission#NEARBY_WIFI_DEVICES} with
 * android:usesPermissionFlags="neverForLocation". If the application does not declare
 * android:usesPermissionFlags="neverForLocation", then it must also have
 * {@link android.Manifest.permission#ACCESS_FINE_LOCATION}.
 * <p>
 * Use {@link #isChannelConstrainedDiscoverySupported()} to determine whether the device
 * supports this feature. If {@link #isChannelConstrainedDiscoverySupported()} return
 * {@code false} then this method will throw {@link java.lang.UnsupportedOperationException UnsupportedOperationException}.
 *
 * @param channel is the channel created at {@link #initialize}
 * This value cannot be {@code null}.
 * @param frequencyMhz is the frequency of the channel to use for peer discovery.
 * @param listener for callbacks on success or failure.

 * This value may be {@code null}.
 */

public void discoverPeersOnSpecificFrequency(@android.annotation.NonNull android.net.wifi.p2p.WifiP2pManager.Channel channel, int frequencyMhz, @android.annotation.Nullable android.net.wifi.p2p.WifiP2pManager.ActionListener listener) { throw new RuntimeException("Stub!"); }

/**
 * Initiate service discovery. A discovery process involves scanning for
 * requested services for the purpose of establishing a connection to a peer
 * that supports an available service.
 *
 * <p> The function call immediately returns after sending a request to start service
 * discovery to the framework. The application is notified of a success or failure to initiate
 * discovery through listener callbacks {@link android.net.wifi.p2p.WifiP2pManager.ActionListener#onSuccess ActionListener.onSuccess} or
 * {@link android.net.wifi.p2p.WifiP2pManager.ActionListener#onFailure ActionListener.onFailure}.
 *
 * <p> The services to be discovered are specified with calls to {@link #addServiceRequest}.
 *
 * <p>The application is notified of the response against the service discovery request
 * through listener callbacks registered by {@link #setServiceResponseListener} or
 * {@link #setDnsSdResponseListeners}, or {@link #setUpnpServiceResponseListener}.
 * <p>
 * If targeting {@link android.os.Build.VERSION_CODES#TIRAMISU} or later, the application must
 * have {@link android.Manifest.permission#NEARBY_WIFI_DEVICES} with
 * android:usesPermissionFlags="neverForLocation". If the application does not declare
 * android:usesPermissionFlags="neverForLocation", then it must also have
 * {@link android.Manifest.permission#ACCESS_FINE_LOCATION}.
 *
 * If targeting an earlier release than {@link android.os.Build.VERSION_CODES#TIRAMISU}, the
 * application must have {@link android.Manifest.permission#ACCESS_FINE_LOCATION}.
 *
 * @param channel is the channel created at {@link #initialize}
 * @param listener for callbacks on success or failure. Can be null.
 */

public void discoverServices(android.net.wifi.p2p.WifiP2pManager.Channel channel, android.net.wifi.p2p.WifiP2pManager.ActionListener listener) { throw new RuntimeException("Stub!"); }

/**
 * Initiate Un-synchronized service discovery (USD) based service discovery. A discovery
 * process involves scanning for requested services for the purpose of establishing a
 * connection to a peer that supports an available service using USD protocol.
 *
 * This method accepts a {@link android.net.wifi.p2p.WifiP2pUsdBasedServiceDiscoveryConfig WifiP2pUsdBasedServiceDiscoveryConfig} object specifying the
 * desired parameters for the service discovery. The configuration object allows to specify
 * either a band or frequency list to scan for service.
 *
 * <p> The function call immediately returns after sending a request to start service
 * discovery to the framework. The application is notified of a success or failure to initiate
 * discovery through listener callbacks {@link android.net.wifi.p2p.WifiP2pManager.ActionListener#onSuccess ActionListener.onSuccess} or
 * {@link android.net.wifi.p2p.WifiP2pManager.ActionListener#onFailure ActionListener.onFailure}.
 *
 * <p> The USD based services to be discovered are specified with calls to
 * {@link #addServiceRequest} with the service request information set through
 * {@link android.net.wifi.p2p.nsd.WifiP2pServiceRequest#WifiP2pServiceRequest(android.net.wifi.p2p.nsd.WifiP2pUsdBasedServiceConfig) WifiP2pServiceRequest.WifiP2pServiceRequest(WifiP2pUsdBasedServiceConfig)}
 *
 * <p>The application is notified of the response against the service discovery request
 * via {@link android.net.wifi.p2p.WifiP2pManager.ServiceResponseListener#onUsdBasedServiceAvailable(android.net.wifi.p2p.WifiP2pDevice,android.net.wifi.p2p.nsd.WifiP2pUsdBasedServiceResponse) ServiceResponseListener.onUsdBasedServiceAvailable(WifiP2pDevice,
 * WifiP2pUsdBasedServiceResponse)} listener callback registered by
 * {@link #setServiceResponseListener(android.net.wifi.p2p.WifiP2pManager.Channel,android.net.wifi.p2p.WifiP2pManager.ServiceResponseListener)} .
 *
 * <p>
 * Use {@link #isWiFiDirectR2Supported()} to determine whether the device supports
 * this feature. If {@link #isWiFiDirectR2Supported()} return {@code false} then
 * this method will throw {@link java.lang.UnsupportedOperationException UnsupportedOperationException}.
 * <p>
 * The application must have {@link android.Manifest.permission#NEARBY_WIFI_DEVICES} with
 * android:usesPermissionFlags="neverForLocation". If the application does not declare
 * android:usesPermissionFlags="neverForLocation", then it must also have
 * {@link android.Manifest.permission#ACCESS_FINE_LOCATION}.
 *
 * @param channel is the channel created at {@link #initialize}
 * This value cannot be {@code null}.
 * @param config is the configuration for this USD based service discovery
 * This value cannot be {@code null}.
 * @param listener for callbacks on success or failure. Can be null.
 */

public void discoverUsdBasedServices(@android.annotation.NonNull android.net.wifi.p2p.WifiP2pManager.Channel channel, @android.annotation.NonNull android.net.wifi.p2p.WifiP2pUsdBasedServiceDiscoveryConfig config, @android.annotation.Nullable android.net.wifi.p2p.WifiP2pManager.ActionListener listener) { throw new RuntimeException("Stub!"); }

/**
 * Get p2p listen state.
 *
 * <p> This state indicates whether p2p listen has started or stopped.
 * The valid value is one of {@link #WIFI_P2P_LISTEN_STOPPED} or
 * {@link #WIFI_P2P_LISTEN_STARTED}.
 *
 * <p> This state is also included in the {@link #ACTION_WIFI_P2P_LISTEN_STATE_CHANGED}
 * broadcast event with extra {@link #EXTRA_LISTEN_STATE}.
 *
 * <p>
 * If targeting {@link android.os.Build.VERSION_CODES#TIRAMISU} or later, the application must
 * have {@link android.Manifest.permission#NEARBY_WIFI_DEVICES} with
 * android:usesPermissionFlags="neverForLocation". If the application does not declare
 * android:usesPermissionFlags="neverForLocation", then it must also have
 * {@link android.Manifest.permission#ACCESS_FINE_LOCATION}.
 *
 * If targeting an earlier release than {@link android.os.Build.VERSION_CODES#TIRAMISU}, the
 * application must have {@link android.Manifest.permission#ACCESS_FINE_LOCATION}.
 *
 * @param c               It is the channel created at {@link #initialize}.
 * This value cannot be {@code null}.
 * @param executor        The executor on which callback will be invoked.
 * This value cannot be {@code null}.
 * Callback and listener events are dispatched through this
 * {@link java.util.concurrent.Executor Executor}, providing an easy way to control which thread is
 * used. To dispatch events through the main thread of your
 * application, you can use
 * {@link android.content.Context#getMainExecutor() Context.getMainExecutor()}.
 * Otherwise, provide an {@link java.util.concurrent.Executor Executor} that dispatches to an appropriate thread.
 * @param resultsCallback A callback that will return listen state
 *                        {@link #WIFI_P2P_LISTEN_STOPPED} or {@link #WIFI_P2P_LISTEN_STARTED}

 * This value cannot be {@code null}.
 */

public void getListenState(@android.annotation.NonNull android.net.wifi.p2p.WifiP2pManager.Channel c, @android.annotation.NonNull java.util.concurrent.Executor executor, @android.annotation.NonNull java.util.function.Consumer<java.lang.Integer> resultsCallback) { throw new RuntimeException("Stub!"); }

/**
 * Return the maximum total length (in bytes) of all Vendor specific information
 * elements (VSIEs) which can be set using the
 * {@link #setVendorElements(android.net.wifi.p2p.WifiP2pManager.Channel,java.util.List,android.net.wifi.p2p.WifiP2pManager.ActionListener)}.
 *
 * The length is calculated adding the payload length + 2 bytes for each VSIE
 * (2 bytes: 1 byte for type and 1 byte for length).
 */

public static int getP2pMaxAllowedVendorElementsLengthBytes() { throw new RuntimeException("Stub!"); }

/**
 * Registers the application with the Wi-Fi framework. This function
 * must be the first to be called before any p2p operations are performed.
 *
 * @param srcContext is the context of the source
 * @param srcLooper is the Looper on which the callbacks are receivied
 * @param listener for callback at loss of framework communication. Can be null.
 * @return Channel instance that is necessary for performing any further p2p operations
 */

public android.net.wifi.p2p.WifiP2pManager.Channel initialize(android.content.Context srcContext, android.os.Looper srcLooper, android.net.wifi.p2p.WifiP2pManager.ChannelListener listener) { throw new RuntimeException("Stub!"); }

/**
 * Check if this device supports discovery limited to a specific frequency or
 * the social channels.
 *
 * Gates whether
 * {@link #discoverPeersOnSpecificFrequency(android.net.wifi.p2p.WifiP2pManager.Channel,int,android.net.wifi.p2p.WifiP2pManager.ActionListener)} and
 * {@link #discoverPeersOnSocialChannels(android.net.wifi.p2p.WifiP2pManager.Channel,android.net.wifi.p2p.WifiP2pManager.ActionListener)}
 * methods are functional on this device.
 *
 * @return {@code true} if supported, {@code false} otherwise.
 */

public boolean isChannelConstrainedDiscoverySupported() { throw new RuntimeException("Stub!"); }

/**
 * Check if this device supports removing clients from a group.
 *
 * Gates whether the
 * {@link #removeClient(android.net.wifi.p2p.WifiP2pManager.Channel,android.net.MacAddress,android.net.wifi.p2p.WifiP2pManager.ActionListener)}
 * method is functional on this device.
 * @return {@code true} if supported, {@code false} otherwise.
 */

public boolean isGroupClientRemovalSupported() { throw new RuntimeException("Stub!"); }

/**
 * Checks whether this device, while being a group client, can discover and deliver the group
 * owner's IPv6 link-local address.
 *
 * <p>If this method returns {@code true} and
 * {@link #connect(android.net.wifi.p2p.WifiP2pManager.Channel,android.net.wifi.p2p.WifiP2pConfig,android.net.wifi.p2p.WifiP2pManager.ActionListener)} method is called with
 * {@link android.net.wifi.p2p.WifiP2pConfig WifiP2pConfig} having
 * {@link android.net.wifi.p2p.WifiP2pConfig#GROUP_CLIENT_IP_PROVISIONING_MODE_IPV6_LINK_LOCAL WifiP2pConfig.GROUP_CLIENT_IP_PROVISIONING_MODE_IPV6_LINK_LOCAL} as the group client
 * IP provisioning mode, then the group owner's IPv6 link-local address will be delivered in the
 * group client via {@link #WIFI_P2P_CONNECTION_CHANGED_ACTION} broadcast intent (i.e, group
 * owner address in {@link #EXTRA_WIFI_P2P_INFO}).
 * If this method returns {@code false}, then IPv6 link-local addresses can still be used, but
 * it is the responsibility of the caller to discover that address in other ways, e.g. using
 * out-of-band communication.
 *
 * @return {@code true} if supported, {@code false} otherwise.
 */

public boolean isGroupOwnerIPv6LinkLocalAddressProvided() { throw new RuntimeException("Stub!"); }

/**
 * Check if this device supports P2P Connection Compatibility Mode(R1/R2 compatibility mode).
 *
 * @return true if this device supports hosting an autonomous Group Owner which allows
 * legacy P2P clients and R2 clients to join the group in PCC Mode and also supports connecting
 * to a Group Owner either using legacy security mode (WPA2-PSK) or R2 mandated security
 * mode(WPA3-SAE) in PCC Mode.
 */

public boolean isPccModeSupported() { throw new RuntimeException("Stub!"); }

/**
 * Check if this device supports setting vendor elements.
 *
 * Gates whether the
 * {@link #setVendorElements(android.net.wifi.p2p.WifiP2pManager.Channel,java.util.List,android.net.wifi.p2p.WifiP2pManager.ActionListener)}
 * method is functional on this device.
 *
 * @return {@code true} if supported, {@code false} otherwise.
 */

public boolean isSetVendorElementsSupported() { throw new RuntimeException("Stub!"); }

/**
 * Check if this device supports Wi-Fi Direct R2 (P2P2).
 *
 * @return true if this device supports Wi-Fi Alliance Wi-Fi Direct R2 (Support for P2P2 IE and
 * establishing connection by using the P2P pairing protocol), false otherwise.
 * For more details, visit <a href="https://www.wi-fi.org/">https://www.wi-fi.org/</a> and
 * search for "Wi-Fi Direct" .
 */

public boolean isWiFiDirectR2Supported() { throw new RuntimeException("Stub!"); }

/**
 * Add a listener to listen to Wi-Fi p2p various changes.
 *
 * @param executor the Executor on which to execute the callbacks.
 * This value cannot be {@code null}.
 * Callback and listener events are dispatched through this
 * {@link java.util.concurrent.Executor Executor}, providing an easy way to control which thread is
 * used. To dispatch events through the main thread of your
 * application, you can use
 * {@link android.content.Context#getMainExecutor() Context.getMainExecutor()}.
 * Otherwise, provide an {@link java.util.concurrent.Executor Executor} that dispatches to an appropriate thread.
 * @param listener listener for the Wi-Fi p2p connection changes.
 * This value cannot be {@code null}.
 * @throws java.lang.SecurityException        if the caller is missing required permissions.
 * @throws java.lang.IllegalArgumentException if incorrect input arguments are provided.
 */

public void registerWifiP2pListener(@android.annotation.NonNull java.util.concurrent.Executor executor, @android.annotation.NonNull android.net.wifi.p2p.WifiP2pManager.WifiP2pListener listener) { throw new RuntimeException("Stub!"); }

/**
 * Remove the client with the MAC address from the group.
 *
 * <p> The function call immediately returns after sending a client removal request
 * to the framework. The application is notified of a success or failure to initiate
 * client removal through listener callbacks {@link android.net.wifi.p2p.WifiP2pManager.ActionListener#onSuccess ActionListener.onSuccess} or
 * {@link android.net.wifi.p2p.WifiP2pManager.ActionListener#onFailure ActionListener.onFailure}.
 *
 * <p> The callbacks are triggered on the thread specified when initializing the
 * {@code channel}, see {@link #initialize}.
 * <p>
 * Use {@link #isGroupClientRemovalSupported()} to determine whether the device supports
 * this feature. If {@link #isGroupClientRemovalSupported()} return {@code false} then this
 * method will throw {@link java.lang.UnsupportedOperationException UnsupportedOperationException}.
 *
 * @param channel is the channel created at {@link #initialize}
 * This value cannot be {@code null}.
 * @param peerAddress MAC address of the client.
 * This value cannot be {@code null}.
 * @param listener for callbacks on success or failure. Can be null.
 */

public void removeClient(@android.annotation.NonNull android.net.wifi.p2p.WifiP2pManager.Channel channel, @android.annotation.NonNull android.net.MacAddress peerAddress, @android.annotation.Nullable android.net.wifi.p2p.WifiP2pManager.ActionListener listener) { throw new RuntimeException("Stub!"); }

/**
 * Remove the external approver for a specific peer.
 *
 * The application must have {@link android.Manifest.permission#NEARBY_WIFI_DEVICES} with
 * android:usesPermissionFlags="neverForLocation". If the application does not declare
 * android:usesPermissionFlags="neverForLocation", then it must also have
 * {@link android.Manifest.permission#ACCESS_FINE_LOCATION}.
 *
 * <br>
 * Requires {@link android.Manifest.permission#MANAGE_WIFI_NETWORK_SELECTION}
 * @param c is the channel created at {@link #initialize(android.content.Context,android.os.Looper,android.net.wifi.p2p.WifiP2pManager.ChannelListener)}.
 * This value cannot be {@code null}.
 * @param deviceAddress the peer which is bound to the external approver.
 * This value cannot be {@code null}.
 * @param listener for callback on success or failure.

 * This value may be {@code null}.
 */

public void removeExternalApprover(@android.annotation.NonNull android.net.wifi.p2p.WifiP2pManager.Channel c, @android.annotation.NonNull android.net.MacAddress deviceAddress, @android.annotation.Nullable android.net.wifi.p2p.WifiP2pManager.ActionListener listener) { throw new RuntimeException("Stub!"); }

/**
 * Remove the current p2p group.
 *
 * <p> The function call immediately returns after sending a group removal request
 * to the framework. The application is notified of a success or failure to initiate
 * group removal through listener callbacks {@link android.net.wifi.p2p.WifiP2pManager.ActionListener#onSuccess ActionListener.onSuccess} or
 * {@link android.net.wifi.p2p.WifiP2pManager.ActionListener#onFailure ActionListener.onFailure}.
 *
 * @param channel is the channel created at {@link #initialize}
 * @param listener for callbacks on success or failure. Can be null.
 */

public void removeGroup(android.net.wifi.p2p.WifiP2pManager.Channel channel, android.net.wifi.p2p.WifiP2pManager.ActionListener listener) { throw new RuntimeException("Stub!"); }

/**
 * Remove a registered local service added with {@link #addLocalService}
 *
 * <p> The function call immediately returns after sending a request to remove a
 * local service to the framework. The application is notified of a success or failure to
 * add service through listener callbacks {@link android.net.wifi.p2p.WifiP2pManager.ActionListener#onSuccess ActionListener.onSuccess} or
 * {@link android.net.wifi.p2p.WifiP2pManager.ActionListener#onFailure ActionListener.onFailure}.
 *
 * @param channel is the channel created at {@link #initialize}
 * @param servInfo is the local service information.
 * @param listener for callbacks on success or failure. Can be null.
 */

public void removeLocalService(android.net.wifi.p2p.WifiP2pManager.Channel channel, android.net.wifi.p2p.nsd.WifiP2pServiceInfo servInfo, android.net.wifi.p2p.WifiP2pManager.ActionListener listener) { throw new RuntimeException("Stub!"); }

/**
 * Remove a specified service discovery request added with {@link #addServiceRequest}
 *
 * <p> The function call immediately returns after sending a request to remove service
 * discovery request to the framework. The application is notified of a success or failure to
 * add service through listener callbacks {@link android.net.wifi.p2p.WifiP2pManager.ActionListener#onSuccess ActionListener.onSuccess} or
 * {@link android.net.wifi.p2p.WifiP2pManager.ActionListener#onFailure ActionListener.onFailure}.
 *
 * @param channel is the channel created at {@link #initialize}
 * @param req is the service discovery request.
 * @param listener for callbacks on success or failure. Can be null.
 */

public void removeServiceRequest(android.net.wifi.p2p.WifiP2pManager.Channel channel, android.net.wifi.p2p.nsd.WifiP2pServiceRequest req, android.net.wifi.p2p.WifiP2pManager.ActionListener listener) { throw new RuntimeException("Stub!"); }

/**
 * Request device connection info.
 *
 * @param channel is the channel created at {@link #initialize}
 * @param listener for callback when connection info is available. Can be null.
 */

public void requestConnectionInfo(android.net.wifi.p2p.WifiP2pManager.Channel channel, android.net.wifi.p2p.WifiP2pManager.ConnectionInfoListener listener) { throw new RuntimeException("Stub!"); }

/**
 * Request Device Info
 *
 * <p> This method provides the device info
 * in the form of a {@link android.net.wifi.p2p.WifiP2pDevice}.
 * Valid {@link android.net.wifi.p2p.WifiP2pDevice} is returned when p2p is enabled.
 * To get information notifications on P2P getting enabled refers
 * {@link #WIFI_P2P_STATE_ENABLED}.
 *
 * <p> This {@link android.net.wifi.p2p.WifiP2pDevice} is returned using the
 * {@link android.net.wifi.p2p.WifiP2pManager.DeviceInfoListener DeviceInfoListener} listener.
 *
 * <p> {@link android.net.wifi.p2p.WifiP2pDevice#deviceAddress} is only available if the caller
 * holds the {@code android.Manifest.permission#LOCAL_MAC_ADDRESS} permission, and holds the
 * anonymized MAC address (02:00:00:00:00:00) otherwise.
 *
 * <p> This information is also included in the {@link #WIFI_P2P_THIS_DEVICE_CHANGED_ACTION}
 * broadcast event with extra {@link #EXTRA_WIFI_P2P_DEVICE}.
 * <p>
 * If targeting {@link android.os.Build.VERSION_CODES#TIRAMISU} or later, the application must
 * have {@link android.Manifest.permission#NEARBY_WIFI_DEVICES} with
 * android:usesPermissionFlags="neverForLocation". If the application does not declare
 * android:usesPermissionFlags="neverForLocation", then it must also have
 * {@link android.Manifest.permission#ACCESS_FINE_LOCATION}.
 *
 * If targeting an earlier release than {@link android.os.Build.VERSION_CODES#TIRAMISU}, the
 * application must have {@link android.Manifest.permission#ACCESS_FINE_LOCATION}.
 *
 * @param c is the channel created at {@link #initialize(android.content.Context,android.os.Looper,android.net.wifi.p2p.WifiP2pManager.ChannelListener)}.
 * This value cannot be {@code null}.
 * @param listener for callback when network info is available.

 * This value cannot be {@code null}.
 */

public void requestDeviceInfo(@android.annotation.NonNull android.net.wifi.p2p.WifiP2pManager.Channel c, @android.annotation.NonNull android.net.wifi.p2p.WifiP2pManager.DeviceInfoListener listener) { throw new RuntimeException("Stub!"); }

/**
 * Get the Device Identity Resolution (DIR) Information.
 * See {@link android.net.wifi.p2p.WifiP2pDirInfo WifiP2pDirInfo} for details
 *
 * Note: The results callback returns null if the device doesn't have any persistent group
 * with device identity key information.
 *
 * <p>
 * Use {@link #isWiFiDirectR2Supported()} to determine whether the device supports
 * this feature. If {@link #isWiFiDirectR2Supported()} return {@code false} then
 * this method will throw {@link java.lang.UnsupportedOperationException UnsupportedOperationException}.
 * <p>
 * The application must have {@link android.Manifest.permission#NEARBY_WIFI_DEVICES} with
 * android:usesPermissionFlags="neverForLocation". If the application does not declare
 * android:usesPermissionFlags="neverForLocation", then it must also have
 * {@link android.Manifest.permission#ACCESS_FINE_LOCATION}.
 *
 * @param c               It is the channel created at {@link #initialize}.
 * This value cannot be {@code null}.
 * @param executor        The executor on which callback will be invoked.
 * This value cannot be {@code null}.
 * Callback and listener events are dispatched through this
 * {@link java.util.concurrent.Executor Executor}, providing an easy way to control which thread is
 * used. To dispatch events through the main thread of your
 * application, you can use
 * {@link android.content.Context#getMainExecutor() Context.getMainExecutor()}.
 * Otherwise, provide an {@link java.util.concurrent.Executor Executor} that dispatches to an appropriate thread.
 * @param callback        An OutcomeReceiver callback for receiving {@link android.net.wifi.p2p.WifiP2pDirInfo WifiP2pDirInfo} via
 *                        {@link android.os.OutcomeReceiver#onResult(java.lang.Object) OutcomeReceiver.onResult(Object)}. This callback will return
 *                        null when DIR info doesn't exist.
 *                        When this API call fails due to permission issues, state machine
 *                        is busy etc., {@link android.os.OutcomeReceiver#onError(java.lang.Throwable) OutcomeReceiver.onError(Throwable)} is called.
 */

public void requestDirInfo(@android.annotation.NonNull android.net.wifi.p2p.WifiP2pManager.Channel c, @android.annotation.NonNull java.util.concurrent.Executor executor, @android.annotation.NonNull android.os.OutcomeReceiver<android.net.wifi.p2p.WifiP2pDirInfo,java.lang.Exception> callback) { throw new RuntimeException("Stub!"); }

/**
 * Request p2p discovery state.
 *
 * <p> This state indicates whether p2p discovery has started or stopped.
 * The valid value is one of {@link #WIFI_P2P_DISCOVERY_STARTED} or
 * {@link #WIFI_P2P_DISCOVERY_STOPPED}. The state is returned using the
 * {@link android.net.wifi.p2p.WifiP2pManager.DiscoveryStateListener DiscoveryStateListener} listener.
 *
 * <p> This state is also included in the {@link #WIFI_P2P_DISCOVERY_CHANGED_ACTION}
 * broadcast event with extra {@link #EXTRA_DISCOVERY_STATE}.
 *
 * @param c is the channel created at {@link #initialize}.
 * This value cannot be {@code null}.
 * @param listener for callback when discovery state is available.

 * This value cannot be {@code null}.
 */

public void requestDiscoveryState(@android.annotation.NonNull android.net.wifi.p2p.WifiP2pManager.Channel c, @android.annotation.NonNull android.net.wifi.p2p.WifiP2pManager.DiscoveryStateListener listener) { throw new RuntimeException("Stub!"); }

/**
 * Request p2p group info.
 * <p>
 * If targeting {@link android.os.Build.VERSION_CODES#TIRAMISU} or later, the application must
 * have {@link android.Manifest.permission#NEARBY_WIFI_DEVICES} with
 * android:usesPermissionFlags="neverForLocation". If the application does not declare
 * android:usesPermissionFlags="neverForLocation", then it must also have
 * {@link android.Manifest.permission#ACCESS_FINE_LOCATION}.
 *
 * If targeting an earlier release than {@link android.os.Build.VERSION_CODES#TIRAMISU}, the
 * application must have {@link android.Manifest.permission#ACCESS_FINE_LOCATION}.
 *
 * @param channel is the channel created at {@link #initialize}
 * @param listener for callback when group info is available. Can be null.
 */

public void requestGroupInfo(android.net.wifi.p2p.WifiP2pManager.Channel channel, android.net.wifi.p2p.WifiP2pManager.GroupInfoListener listener) { throw new RuntimeException("Stub!"); }

/**
 * Request network info.
 *
 * <p> This method provides the network info in the form of a {@link android.net.NetworkInfo}.
 * {@link android.net.NetworkInfo#isAvailable()} indicates the p2p availability and
 * {@link android.net.NetworkInfo#getDetailedState()} reports the current fine-grained state
 * of the network. This {@link android.net.NetworkInfo} is returned using the
 * {@link android.net.wifi.p2p.WifiP2pManager.NetworkInfoListener NetworkInfoListener} listener.
 *
 * <p> This information is also included in the {@link #WIFI_P2P_CONNECTION_CHANGED_ACTION}
 * broadcast event with extra {@link #EXTRA_NETWORK_INFO}.
 *
 * @param c is the channel created at {@link #initialize}.
 * This value cannot be {@code null}.
 * @param listener for callback when network info is available.

 * This value cannot be {@code null}.
 */

public void requestNetworkInfo(@android.annotation.NonNull android.net.wifi.p2p.WifiP2pManager.Channel c, @android.annotation.NonNull android.net.wifi.p2p.WifiP2pManager.NetworkInfoListener listener) { throw new RuntimeException("Stub!"); }

/**
 * Request p2p enabled state.
 *
 * <p> This state indicates whether Wi-Fi p2p is enabled or disabled.
 * The valid value is one of {@link #WIFI_P2P_STATE_DISABLED} or
 * {@link #WIFI_P2P_STATE_ENABLED}. The state is returned using the
 * {@link android.net.wifi.p2p.WifiP2pManager.P2pStateListener P2pStateListener} listener.
 *
 * <p> This state is also included in the {@link #WIFI_P2P_STATE_CHANGED_ACTION}
 * broadcast event with extra {@link #EXTRA_WIFI_STATE}.
 *
 * @param c is the channel created at {@link #initialize}.
 * This value cannot be {@code null}.
 * @param listener for callback when p2p state is available.

 * This value cannot be {@code null}.
 */

public void requestP2pState(@android.annotation.NonNull android.net.wifi.p2p.WifiP2pManager.Channel c, @android.annotation.NonNull android.net.wifi.p2p.WifiP2pManager.P2pStateListener listener) { throw new RuntimeException("Stub!"); }

/**
 * Request the current list of peers.
 * <p>
 * If targeting {@link android.os.Build.VERSION_CODES#TIRAMISU} or later, the application must
 * have {@link android.Manifest.permission#NEARBY_WIFI_DEVICES} with
 * android:usesPermissionFlags="neverForLocation". If the application does not declare
 * android:usesPermissionFlags="neverForLocation", then it must also have
 * {@link android.Manifest.permission#ACCESS_FINE_LOCATION}.
 *
 * If targeting an earlier release than {@link android.os.Build.VERSION_CODES#TIRAMISU}, the
 * application must have {@link android.Manifest.permission#ACCESS_FINE_LOCATION}.
 *
 * @param channel is the channel created at {@link #initialize}
 * @param listener for callback when peer list is available. Can be null.
 */

public void requestPeers(android.net.wifi.p2p.WifiP2pManager.Channel channel, android.net.wifi.p2p.WifiP2pManager.PeerListListener listener) { throw new RuntimeException("Stub!"); }

/**
 * Set the result for the incoming request from a specific peer.
 *
 * The application must have {@link android.Manifest.permission#NEARBY_WIFI_DEVICES} with
 * android:usesPermissionFlags="neverForLocation". If the application does not declare
 * android:usesPermissionFlags="neverForLocation", then it must also have
 * {@link android.Manifest.permission#ACCESS_FINE_LOCATION}.
 *
 * <br>
 * Requires {@link android.Manifest.permission#MANAGE_WIFI_NETWORK_SELECTION}
 * @param c is the channel created at {@link #initialize(android.content.Context,android.os.Looper,android.net.wifi.p2p.WifiP2pManager.ChannelListener)}.
 * This value cannot be {@code null}.
 * @param deviceAddress the peer which is bound to the external approver.
 * This value cannot be {@code null}.
 * @param result the response for the incoming request.
 * Value is {@link android.net.wifi.p2p.WifiP2pManager#CONNECTION_REQUEST_ACCEPT}, {@link android.net.wifi.p2p.WifiP2pManager#CONNECTION_REQUEST_REJECT}, {@link android.net.wifi.p2p.WifiP2pManager#CONNECTION_REQUEST_DEFER_TO_SERVICE}, or {@link android.net.wifi.p2p.WifiP2pManager#CONNECTION_REQUEST_DEFER_SHOW_PIN_TO_SERVICE}
 * @param listener for callback on success or failure.

 * This value may be {@code null}.
 */

public void setConnectionRequestResult(@android.annotation.NonNull android.net.wifi.p2p.WifiP2pManager.Channel c, @android.annotation.NonNull android.net.MacAddress deviceAddress, int result, @android.annotation.Nullable android.net.wifi.p2p.WifiP2pManager.ActionListener listener) { throw new RuntimeException("Stub!"); }

/**
 * Set the result with PIN for the incoming request from a specific peer.
 *
 * The application must have {@link android.Manifest.permission#NEARBY_WIFI_DEVICES} with
 * android:usesPermissionFlags="neverForLocation". If the application does not declare
 * android:usesPermissionFlags="neverForLocation", then it must also have
 * {@link android.Manifest.permission#ACCESS_FINE_LOCATION}.
 *
 * <br>
 * Requires {@link android.Manifest.permission#MANAGE_WIFI_NETWORK_SELECTION}
 * @param c is the channel created at {@link #initialize(android.content.Context,android.os.Looper,android.net.wifi.p2p.WifiP2pManager.ChannelListener)}.
 * This value cannot be {@code null}.
 * @param deviceAddress the peer which is bound to the external approver.
 * This value cannot be {@code null}.
 * @param result the response for the incoming request.
 * Value is {@link android.net.wifi.p2p.WifiP2pManager#CONNECTION_REQUEST_ACCEPT}, {@link android.net.wifi.p2p.WifiP2pManager#CONNECTION_REQUEST_REJECT}, {@link android.net.wifi.p2p.WifiP2pManager#CONNECTION_REQUEST_DEFER_TO_SERVICE}, or {@link android.net.wifi.p2p.WifiP2pManager#CONNECTION_REQUEST_DEFER_SHOW_PIN_TO_SERVICE}
 * @param pin the PIN for the incoming request.
 * This value may be {@code null}.
 * @param listener for callback on success or failure.

 * This value may be {@code null}.
 */

public void setConnectionRequestResult(@android.annotation.NonNull android.net.wifi.p2p.WifiP2pManager.Channel c, @android.annotation.NonNull android.net.MacAddress deviceAddress, int result, @android.annotation.Nullable java.lang.String pin, @android.annotation.Nullable android.net.wifi.p2p.WifiP2pManager.ActionListener listener) { throw new RuntimeException("Stub!"); }

/**
 * Register a callback to be invoked on receiving Bonjour service discovery
 * response.
 *
 * <p> see {@link #discoverServices} for the detail.
 *
 * @param channel
 * @param servListener is for listening to a Bonjour service response
 * @param txtListener is for listening to a Bonjour TXT record response
 */

public void setDnsSdResponseListeners(android.net.wifi.p2p.WifiP2pManager.Channel channel, android.net.wifi.p2p.WifiP2pManager.DnsSdServiceResponseListener servListener, android.net.wifi.p2p.WifiP2pManager.DnsSdTxtRecordListener txtListener) { throw new RuntimeException("Stub!"); }

/**
 * Register a callback to be invoked on receiving service discovery response.
 * Used only for vendor specific protocol right now. For Bonjour or Upnp, use
 * {@link #setDnsSdResponseListeners} or {@link #setUpnpServiceResponseListener}
 * respectively.
 *
 * <p> see {@link #discoverServices} for the detail.
 *
 * @param channel is the channel created at {@link #initialize}
 * @param listener for callbacks on receiving service discovery response.
 */

public void setServiceResponseListener(android.net.wifi.p2p.WifiP2pManager.Channel channel, android.net.wifi.p2p.WifiP2pManager.ServiceResponseListener listener) { throw new RuntimeException("Stub!"); }

/**
 * Register a callback to be invoked on receiving upnp service discovery
 * response.
 *
 * <p> see {@link #discoverServices} for the detail.
 *
 * @param channel is the channel created at {@link #initialize}
 * @param listener for callbacks on receiving service discovery response.
 */

public void setUpnpServiceResponseListener(android.net.wifi.p2p.WifiP2pManager.Channel channel, android.net.wifi.p2p.WifiP2pManager.UpnpServiceResponseListener listener) { throw new RuntimeException("Stub!"); }

/**
 * Set/Clear vendor specific information elements (VSIEs) to be published during
 * Wi-Fi Direct (P2P) discovery.
 *
 * Once {@link android.net.wifi.p2p.WifiP2pManager.Channel#close() Channel.close()} is called, the vendor information elements will be cleared from
 * framework. The information element format is defined in the IEEE 802.11-2016 spec
 * Table 9-77.
 * <p>
 * To clear the previously set vendor elements, call this API with an empty List.
 * <p>
 * The maximum accumulated length of all VSIEs must be before the limit specified by
 * {@link #getP2pMaxAllowedVendorElementsLengthBytes()}.
 * <p>
 * To publish vendor elements, this API should be called before peer discovery API, ex.
 * {@link #discoverPeers(android.net.wifi.p2p.WifiP2pManager.Channel,android.net.wifi.p2p.WifiP2pManager.ActionListener)}.
 * <p>
 * Use {@link #isSetVendorElementsSupported()} to determine whether the device supports
 * this feature. If {@link #isSetVendorElementsSupported()} return {@code false} then
 * this method will throw {@link java.lang.UnsupportedOperationException UnsupportedOperationException}.
 *
 * <br>
 * Requires {@link android.Manifest.permission#NEARBY_WIFI_DEVICES} and {@link android.Manifest.permission#OVERRIDE_WIFI_CONFIG}
 * @param c is the channel created at {@link #initialize(android.content.Context,android.os.Looper,android.net.wifi.p2p.WifiP2pManager.ChannelListener)}.
 * This value cannot be {@code null}.
 * @param vendorElements application information as vendor-specific information elements.
 * This value cannot be {@code null}.
 * @param listener for callback when network info is available.

 * This value may be {@code null}.
 */

public void setVendorElements(@android.annotation.NonNull android.net.wifi.p2p.WifiP2pManager.Channel c, @android.annotation.NonNull java.util.List<android.net.wifi.ScanResult.InformationElement> vendorElements, @android.annotation.Nullable android.net.wifi.p2p.WifiP2pManager.ActionListener listener) { throw new RuntimeException("Stub!"); }

/**
 * Set Wifi Display information.
 *
 * <br>
 * Requires {@link android.Manifest.permission#CONFIGURE_WIFI_DISPLAY}
 * @param channel is the channel created at {@link #initialize}
 * This value cannot be {@code null}.
 * @param wfdInfo the Wifi Display information to set
 * This value cannot be {@code null}.
 * @param listener for callbacks on success or failure. Can be null.
 */

public void setWfdInfo(@android.annotation.NonNull android.net.wifi.p2p.WifiP2pManager.Channel channel, @android.annotation.NonNull android.net.wifi.p2p.WifiP2pWfdInfo wfdInfo, @android.annotation.Nullable android.net.wifi.p2p.WifiP2pManager.ActionListener listener) { throw new RuntimeException("Stub!"); }

/**
 * Force p2p to enter listen state.
 *
 * When this API is called, this device will periodically enter LISTENING state until
 * {@link #stopListening(android.net.wifi.p2p.WifiP2pManager.Channel,android.net.wifi.p2p.WifiP2pManager.ActionListener)} or
 * {@link #stopPeerDiscovery(android.net.wifi.p2p.WifiP2pManager.Channel,android.net.wifi.p2p.WifiP2pManager.ActionListener)} are called.
 * While in LISTENING state, this device will dwell at its social channel and respond
 * to probe requests from other Wi-Fi Direct peers.
 * <p>
 * If targeting {@link android.os.Build.VERSION_CODES#TIRAMISU} or later, the application must
 * have {@link android.Manifest.permission#NEARBY_WIFI_DEVICES} with
 * android:usesPermissionFlags="neverForLocation". If the application does not declare
 * android:usesPermissionFlags="neverForLocation", then it must also have
 * {@link android.Manifest.permission#ACCESS_FINE_LOCATION}.
 *
 * If targeting an earlier release than {@link android.os.Build.VERSION_CODES#TIRAMISU}, the
 * application must have {@link android.Manifest.permission#ACCESS_FINE_LOCATION}.
 * @param channel is the channel created at
 *    {@link #initialize(android.content.Context,android.os.Looper,android.net.wifi.p2p.WifiP2pManager.ChannelListener)}
 * This value cannot be {@code null}.
 * @param listener for callbacks on success or failure.

 * This value may be {@code null}.
 */

public void startListening(@android.annotation.NonNull android.net.wifi.p2p.WifiP2pManager.Channel channel, @android.annotation.Nullable android.net.wifi.p2p.WifiP2pManager.ActionListener listener) { throw new RuntimeException("Stub!"); }

/**
 * Initiate peer discovery. A discovery process involves scanning for available Wi-Fi peers
 * for the purpose of establishing a connection. See {@link #discoverPeers(android.net.wifi.p2p.WifiP2pManager.Channel,android.net.wifi.p2p.WifiP2pManager.ActionListener)} for more details.
 *
 * This method accepts a {@link android.net.wifi.p2p.WifiP2pDiscoveryConfig WifiP2pDiscoveryConfig} object specifying the desired
 * parameters for the peer discovery. The configuration object allows the specification of the
 * scan type (ex. FULL, SOCIAL) and the inclusion of vendor-specific configuration data.
 *
 * The application must have {@link android.Manifest.permission#NEARBY_WIFI_DEVICES} with
 * android:usesPermissionFlags="neverForLocation". If the application does not declare
 * android:usesPermissionFlags="neverForLocation", then it must also have
 * {@link android.Manifest.permission#ACCESS_FINE_LOCATION}.
 *
 * @param channel is the channel created at {@link #initialize}
 * This value cannot be {@code null}.
 * @param config is the configuration for this peer discovery
 * This value cannot be {@code null}.
 * @param listener for callbacks on success or failure.

 * This value may be {@code null}.
 */

public void startPeerDiscovery(@android.annotation.NonNull android.net.wifi.p2p.WifiP2pManager.Channel channel, @android.annotation.NonNull android.net.wifi.p2p.WifiP2pDiscoveryConfig config, @android.annotation.Nullable android.net.wifi.p2p.WifiP2pManager.ActionListener listener) { throw new RuntimeException("Stub!"); }

/**
 * Start a service discovery advertisement using Un-synchronized service discovery (USD).
 * Once {@link #startUsdBasedLocalServiceAdvertisement(android.net.wifi.p2p.WifiP2pManager.Channel,android.net.wifi.p2p.nsd.WifiP2pServiceInfo,android.net.wifi.p2p.WifiP2pUsdBasedLocalServiceAdvertisementConfig,android.net.wifi.p2p.WifiP2pManager.ActionListener)} is called, the device will
 * go to the channel frequency requested via
 * {@link android.net.wifi.p2p.WifiP2pUsdBasedLocalServiceAdvertisementConfig WifiP2pUsdBasedLocalServiceAdvertisementConfig} and responds to a service discovery
 * request from a peer.
 *
 * <p> The service information is set through
 * {@link android.net.wifi.p2p.nsd.WifiP2pServiceInfo#WifiP2pServiceInfo(android.net.wifi.p2p.nsd.WifiP2pUsdBasedServiceConfig) WifiP2pServiceInfo.WifiP2pServiceInfo(WifiP2pUsdBasedServiceConfig)}
 *
 * <p> The function call immediately returns after sending a request to start the service
 * advertisement to the framework. The application is notified of a success or failure to
 * start service advertisement through listener callbacks {@link android.net.wifi.p2p.WifiP2pManager.ActionListener#onSuccess ActionListener.onSuccess} or
 * {@link android.net.wifi.p2p.WifiP2pManager.ActionListener#onFailure ActionListener.onFailure}.
 *
 * <p>The service information can be cleared with calls to
 *  {@link #removeLocalService} or {@link #clearLocalServices}.
 * <p>
 * Use {@link #isWiFiDirectR2Supported()} to determine whether the device supports
 * this feature. If {@link #isWiFiDirectR2Supported()} return {@code false} then
 * this method will throw {@link java.lang.UnsupportedOperationException UnsupportedOperationException}.
 * <p>
 * The application must have {@link android.Manifest.permission#NEARBY_WIFI_DEVICES} with
 * android:usesPermissionFlags="neverForLocation". If the application does not declare
 * android:usesPermissionFlags="neverForLocation", then it must also have
 * {@link android.Manifest.permission#ACCESS_FINE_LOCATION}.
 *
 * @param channel is the channel created at {@link #initialize}
 * This value cannot be {@code null}.
 * @param servInfo is a local service information.
 * This value cannot be {@code null}.
 * @param config is the configuration for this service discovery advertisement.
 * This value cannot be {@code null}.
 * @param listener for callbacks on success or failure. Can be null.
 */

public void startUsdBasedLocalServiceAdvertisement(@android.annotation.NonNull android.net.wifi.p2p.WifiP2pManager.Channel channel, @android.annotation.NonNull android.net.wifi.p2p.nsd.WifiP2pServiceInfo servInfo, @android.annotation.NonNull android.net.wifi.p2p.WifiP2pUsdBasedLocalServiceAdvertisementConfig config, @android.annotation.Nullable android.net.wifi.p2p.WifiP2pManager.ActionListener listener) { throw new RuntimeException("Stub!"); }

/**
 * Force p2p to exit listen state.
 *
 * When this API is called, this device will stop entering LISTENING state periodically
 * which is triggered by {@link #startListening(android.net.wifi.p2p.WifiP2pManager.Channel,android.net.wifi.p2p.WifiP2pManager.ActionListener)}.
 * If there are running peer discovery which is triggered by
 * {@link #discoverPeers(android.net.wifi.p2p.WifiP2pManager.Channel,android.net.wifi.p2p.WifiP2pManager.ActionListener)} or running service discovery which is
 * triggered by {@link #discoverServices(android.net.wifi.p2p.WifiP2pManager.Channel,android.net.wifi.p2p.WifiP2pManager.ActionListener)}, they will be stopped
 * as well.
 *
 * @param channel is the channel created at
 *    {@link #initialize(android.content.Context,android.os.Looper,android.net.wifi.p2p.WifiP2pManager.ChannelListener)}
 * This value cannot be {@code null}.
 * @param listener for callbacks on success or failure.

 * This value may be {@code null}.
 */

public void stopListening(@android.annotation.NonNull android.net.wifi.p2p.WifiP2pManager.Channel channel, @android.annotation.Nullable android.net.wifi.p2p.WifiP2pManager.ActionListener listener) { throw new RuntimeException("Stub!"); }

/**
 * Stop an ongoing peer discovery
 *
 * <p> The function call immediately returns after sending a stop request
 * to the framework. The application is notified of a success or failure to initiate
 * stop through listener callbacks {@link android.net.wifi.p2p.WifiP2pManager.ActionListener#onSuccess ActionListener.onSuccess} or
 * {@link android.net.wifi.p2p.WifiP2pManager.ActionListener#onFailure ActionListener.onFailure}.
 *
 * <p> If P2P Group is in the process of being created, this call will fail (report failure via
 * {@code listener}. The applicantion should listen to
 * {@link #WIFI_P2P_CONNECTION_CHANGED_ACTION} to ensure the state is not
 * {@link android.net.NetworkInfo.State#CONNECTING} and repeat calling when the state changes.
 *
 * @param channel is the channel created at {@link #initialize}
 * @param listener for callbacks on success or failure. Can be null.
 */

public void stopPeerDiscovery(android.net.wifi.p2p.WifiP2pManager.Channel channel, android.net.wifi.p2p.WifiP2pManager.ActionListener listener) { throw new RuntimeException("Stub!"); }

/**
 * Remove a listener added using
 * {@link #registerWifiP2pListener(java.util.concurrent.Executor,android.net.wifi.p2p.WifiP2pManager.WifiP2pListener)}
 *
 * @param listener the listener to be removed.
 * This value cannot be {@code null}.
 * @throws java.lang.IllegalArgumentException if incorrect input arguments are provided.
 */

public void unregisterWifiP2pListener(@android.annotation.NonNull android.net.wifi.p2p.WifiP2pManager.WifiP2pListener listener) { throw new RuntimeException("Stub!"); }

/**
 * Validate the Device Identity Resolution (DIR) Information of a P2P device.
 * See {@link android.net.wifi.p2p.WifiP2pDirInfo WifiP2pDirInfo} for details.
 * Framework takes the {@link android.net.wifi.p2p.WifiP2pDirInfo WifiP2pDirInfo} and derives a set of Tag values based on
 * the cached Device Identity Keys (DevIK) of all paired peers saved in the device.
 * If a derived Tag value matches the Tag value received in the {@link android.net.wifi.p2p.WifiP2pDirInfo WifiP2pDirInfo}, the
 * device is identified as a paired peer and returns true.
 *
 * <p>
 * Use {@link #isWiFiDirectR2Supported()} to determine whether the device supports
 * this feature. If {@link #isWiFiDirectR2Supported()} return {@code false} then
 * this method will throw {@link java.lang.UnsupportedOperationException UnsupportedOperationException}.
 * <p>
 * The application must have {@link android.Manifest.permission#NEARBY_WIFI_DEVICES} with
 * android:usesPermissionFlags="neverForLocation". If the application does not declare
 * android:usesPermissionFlags="neverForLocation", then it must also have
 * {@link android.Manifest.permission#ACCESS_FINE_LOCATION}.
 *
 * @param c               It is the channel created at {@link #initialize}.
 * This value cannot be {@code null}.
 * @param dirInfo         {@link android.net.wifi.p2p.WifiP2pDirInfo WifiP2pDirInfo} to validate.
 * This value cannot be {@code null}.
 * @param executor        The executor on which callback will be invoked.
 * This value cannot be {@code null}.
 * Callback and listener events are dispatched through this
 * {@link java.util.concurrent.Executor Executor}, providing an easy way to control which thread is
 * used. To dispatch events through the main thread of your
 * application, you can use
 * {@link android.content.Context#getMainExecutor() Context.getMainExecutor()}.
 * Otherwise, provide an {@link java.util.concurrent.Executor Executor} that dispatches to an appropriate thread.
 * @param callback        An OutcomeReceiver callback for receiving the result via
 *                        {@link android.os.OutcomeReceiver#onResult(java.lang.Object) OutcomeReceiver.onResult(Object)} indicating whether the DIR
 *                        info of P2P device is of a paired device. {code true} for paired,
 *                        {@code false} for not paired.
 *                        When this API call fails due to permission issues, state machine
 *                        is busy etc., {@link android.os.OutcomeReceiver#onError(java.lang.Throwable) OutcomeReceiver.onError(Throwable)} is called.

 * This value cannot be {@code null}.
 */

public void validateDirInfo(@android.annotation.NonNull android.net.wifi.p2p.WifiP2pManager.Channel c, @android.annotation.NonNull android.net.wifi.p2p.WifiP2pDirInfo dirInfo, @android.annotation.NonNull java.util.concurrent.Executor executor, @android.annotation.NonNull android.os.OutcomeReceiver<java.lang.Boolean,java.lang.Exception> callback) { throw new RuntimeException("Stub!"); }

/**
 * Broadcast intent action indicating that peer listen has either started or stopped.
 * One extra {@link #EXTRA_LISTEN_STATE} indicates whether listen has started or stopped.
 */

@android.annotation.SdkConstant(android.annotation.SdkConstant.SdkConstantType.BROADCAST_INTENT_ACTION) public static final java.lang.String ACTION_WIFI_P2P_LISTEN_STATE_CHANGED = "android.net.wifi.p2p.action.WIFI_P2P_LISTEN_STATE_CHANGED";

/**
 * Broadcast intent action indicating whether or not current connecting
 * request is accepted.
 *
 * The connecting request is initiated by
 * {@link #connect(android.net.wifi.p2p.WifiP2pManager.Channel,android.net.wifi.p2p.WifiP2pConfig,android.net.wifi.p2p.WifiP2pManager.ActionListener)}.
 * <p>The {@link #EXTRA_REQUEST_RESPONSE} extra indicates whether or not current
 * request is accepted or rejected.
 * <p>The {@link #EXTRA_REQUEST_CONFIG} extra indicates the responsed configuration.
 */

public static final java.lang.String ACTION_WIFI_P2P_REQUEST_RESPONSE_CHANGED = "android.net.wifi.p2p.action.WIFI_P2P_REQUEST_RESPONSE_CHANGED";

/**
 * Passed with {@link android.net.wifi.p2p.WifiP2pManager.ActionListener#onFailure ActionListener.onFailure}.
 * Indicates that the operation failed because the framework is busy and
 * unable to service the request
 */

public static final int BUSY = 2; // 0x2

/**
 * Accept the incoming request.
 *
 * Used in {@link #setConnectionRequestResult(android.net.wifi.p2p.WifiP2pManager.Channel,android.net.MacAddress,int,android.net.wifi.p2p.WifiP2pManager.ActionListener)}.
 */

public static final int CONNECTION_REQUEST_ACCEPT = 0; // 0x0

/**
 * Defer the PIN display to the Wi-Fi service (which will display a dialog to the user).
 *
 * Used in {@link #setConnectionRequestResult(android.net.wifi.p2p.WifiP2pManager.Channel,android.net.MacAddress,int,android.net.wifi.p2p.WifiP2pManager.ActionListener)}.
 */

public static final int CONNECTION_REQUEST_DEFER_SHOW_PIN_TO_SERVICE = 3; // 0x3

/**
 * Defer the decision back to the Wi-Fi service (which will display a dialog to the user).
 *
 * Used in {@link #setConnectionRequestResult(android.net.wifi.p2p.WifiP2pManager.Channel,android.net.MacAddress,int,android.net.wifi.p2p.WifiP2pManager.ActionListener)}.
 */

public static final int CONNECTION_REQUEST_DEFER_TO_SERVICE = 2; // 0x2

/**
 * Reject the incoming request.
 *
 * Used in {@link #setConnectionRequestResult(android.net.wifi.p2p.WifiP2pManager.Channel,android.net.MacAddress,int,android.net.wifi.p2p.WifiP2pManager.ActionListener)}.
 */

public static final int CONNECTION_REQUEST_REJECT = 1; // 0x1

/**
 * Passed with {@link android.net.wifi.p2p.WifiP2pManager.ActionListener#onFailure ActionListener.onFailure}.
 * Indicates that the operation failed due to an internal error.
 */

public static final int ERROR = 0; // 0x0

/**
 * The lookup key for an int that indicates whether p2p discovery has started or stopped.
 * Retrieve it with {@link android.content.Intent#getIntExtra(String,int)}.
 *
 * @see #WIFI_P2P_DISCOVERY_STARTED
 * @see #WIFI_P2P_DISCOVERY_STOPPED
 */

public static final java.lang.String EXTRA_DISCOVERY_STATE = "discoveryState";

/**
 * The lookup key for an int that indicates whether p2p listen has started or stopped.
 * Retrieve it with {@link android.content.Intent#getIntExtra(String,int)}.
 *
 * @see #WIFI_P2P_LISTEN_STARTED
 * @see #WIFI_P2P_LISTEN_STOPPED
 */

public static final java.lang.String EXTRA_LISTEN_STATE = "android.net.wifi.p2p.extra.LISTEN_STATE";

/**
 * The lookup key for a {@link android.net.NetworkInfo} object associated with the
 * p2p network. Retrieve with
 * {@link android.content.Intent#getParcelableExtra(String)}.
 */

public static final java.lang.String EXTRA_NETWORK_INFO = "networkInfo";

/**
 * The lookup key for a {@link android.net.wifi.p2p.WifiP2pDeviceList} object representing
 * the new peer list when {@link #WIFI_P2P_PEERS_CHANGED_ACTION} broadcast is sent.
 *
 * <p>Retrieve with {@link android.content.Intent#getParcelableExtra(String)}.
 */

public static final java.lang.String EXTRA_P2P_DEVICE_LIST = "wifiP2pDeviceList";

/**
 * The lookup key for the {@link android.net.wifi.p2p.WifiP2pConfig WifiP2pConfig} object of a request.
 */

public static final java.lang.String EXTRA_REQUEST_CONFIG = "android.net.wifi.p2p.extra.REQUEST_CONFIG";

/**
 * The lookup key for the result of a request, true if accepted, false otherwise.
 */

public static final java.lang.String EXTRA_REQUEST_RESPONSE = "android.net.wifi.p2p.extra.REQUEST_RESPONSE";

/**
 * The lookup key for a {@link android.net.wifi.p2p.WifiP2pDevice} object
 * Retrieve with {@link android.content.Intent#getParcelableExtra(String)}.
 */

public static final java.lang.String EXTRA_WIFI_P2P_DEVICE = "wifiP2pDevice";

/**
 * The lookup key for a {@link android.net.wifi.p2p.WifiP2pGroup} object
 * associated with the p2p network. Retrieve with
 * {@link android.content.Intent#getParcelableExtra(String)}.
 */

public static final java.lang.String EXTRA_WIFI_P2P_GROUP = "p2pGroupInfo";

/**
 * The lookup key for a {@link android.net.wifi.p2p.WifiP2pInfo} object
 * Retrieve with {@link android.content.Intent#getParcelableExtra(String)}.
 */

public static final java.lang.String EXTRA_WIFI_P2P_INFO = "wifiP2pInfo";

/**
 * The lookup key for an int that indicates whether Wi-Fi p2p is enabled or disabled.
 * Retrieve it with {@link android.content.Intent#getIntExtra(String,int)}.
 *
 * @see #WIFI_P2P_STATE_DISABLED
 * @see #WIFI_P2P_STATE_ENABLED
 */

public static final java.lang.String EXTRA_WIFI_STATE = "wifi_p2p_state";

/**
 * P2p group creation failed because the connection has been cancelled.
 * Used in {@link android.net.wifi.p2p.WifiP2pManager.WifiP2pListener#onGroupCreationFailed(int reason) WifiP2pListener.onGroupCreationFailed(int reason)}.
 */

public static final int GROUP_CREATION_FAILURE_REASON_CONNECTION_CANCELLED = 0; // 0x0

/**
 * P2p group creation failed because the group has been removed.
 * Used in {@link android.net.wifi.p2p.WifiP2pManager.WifiP2pListener#onGroupCreationFailed(int reason) WifiP2pListener.onGroupCreationFailed(int reason)}.
 */

public static final int GROUP_CREATION_FAILURE_REASON_GROUP_REMOVED = 4; // 0x4

/**
 * P2p group creation failed because invitation has failed.
 * Used in {@link android.net.wifi.p2p.WifiP2pManager.WifiP2pListener#onGroupCreationFailed(int reason) WifiP2pListener.onGroupCreationFailed(int reason)}.
 */

public static final int GROUP_CREATION_FAILURE_REASON_INVITATION_FAILED = 5; // 0x5

/**
 * P2p group creation failed because provision discovery has failed.
 * Used in {@link android.net.wifi.p2p.WifiP2pManager.WifiP2pListener#onGroupCreationFailed(int reason) WifiP2pListener.onGroupCreationFailed(int reason)}.
 */

public static final int GROUP_CREATION_FAILURE_REASON_PROVISION_DISCOVERY_FAILED = 3; // 0x3

/**
 * P2p group creation failed because it has timed out.
 * Used in {@link android.net.wifi.p2p.WifiP2pManager.WifiP2pListener#onGroupCreationFailed(int reason) WifiP2pListener.onGroupCreationFailed(int reason)}.
 */

public static final int GROUP_CREATION_FAILURE_REASON_TIMED_OUT = 1; // 0x1

/**
 * P2p group creation failed because user has rejected.
 * Used in {@link android.net.wifi.p2p.WifiP2pManager.WifiP2pListener#onGroupCreationFailed(int reason) WifiP2pListener.onGroupCreationFailed(int reason)}.
 */

public static final int GROUP_CREATION_FAILURE_REASON_USER_REJECTED = 2; // 0x2

/**
 * Passed with {@link android.net.wifi.p2p.WifiP2pManager.ActionListener#onFailure ActionListener.onFailure}.
 * Indicates that the operation failed due to calling app doesn't have permission to call the
 * API.
 */

public static final int NO_PERMISSION = 4; // 0x4

/**
 * Passed with {@link android.net.wifi.p2p.WifiP2pManager.ActionListener#onFailure ActionListener.onFailure}.
 * Indicates that the {@link #discoverServices} failed because no service
 * requests are added. Use {@link #addServiceRequest} to add a service
 * request.
 */

public static final int NO_SERVICE_REQUESTS = 3; // 0x3

/**
 * Passed with {@link android.net.wifi.p2p.WifiP2pManager.ActionListener#onFailure ActionListener.onFailure}.
 * Indicates that the operation failed because p2p is unsupported on the device.
 */

public static final int P2P_UNSUPPORTED = 1; // 0x1

/**
 * Broadcast intent action indicating that the state of Wi-Fi p2p connectivity
 * has changed. One extra {@link #EXTRA_WIFI_P2P_INFO} provides the p2p connection info in
 * the form of a {@link android.net.wifi.p2p.WifiP2pInfo WifiP2pInfo} object. Another extra {@link #EXTRA_NETWORK_INFO} provides
 * the network info in the form of a {@link android.net.NetworkInfo}. A third extra provides
 * the details of the group and may contain a {@code null}.
 *
 * All of these permissions are required to receive this broadcast:
 * {@link android.Manifest.permission#ACCESS_WIFI_STATE} and either
 * {@link android.Manifest.permission#ACCESS_FINE_LOCATION} or
 * {@link android.Manifest.permission#NEARBY_WIFI_DEVICES}
 *
 * @see #EXTRA_WIFI_P2P_INFO
 * @see #EXTRA_NETWORK_INFO
 * @see #EXTRA_WIFI_P2P_GROUP
 */

@android.annotation.SdkConstant(android.annotation.SdkConstant.SdkConstantType.BROADCAST_INTENT_ACTION) public static final java.lang.String WIFI_P2P_CONNECTION_CHANGED_ACTION = "android.net.wifi.p2p.CONNECTION_STATE_CHANGE";

/**
 * Broadcast intent action indicating that peer discovery has either started or stopped.
 * One extra {@link #EXTRA_DISCOVERY_STATE} indicates whether discovery has started
 * or stopped.
 *
 * <p>Note that discovery will be stopped during a connection setup. If the application tries
 * to re-initiate discovery during this time, it can fail.
 */

@android.annotation.SdkConstant(android.annotation.SdkConstant.SdkConstantType.BROADCAST_INTENT_ACTION) public static final java.lang.String WIFI_P2P_DISCOVERY_CHANGED_ACTION = "android.net.wifi.p2p.DISCOVERY_STATE_CHANGE";

/**
 * p2p discovery has started
 *
 * @see #WIFI_P2P_DISCOVERY_CHANGED_ACTION
 */

public static final int WIFI_P2P_DISCOVERY_STARTED = 2; // 0x2

/**
 * p2p discovery has stopped
 *
 * @see #WIFI_P2P_DISCOVERY_CHANGED_ACTION
 */

public static final int WIFI_P2P_DISCOVERY_STOPPED = 1; // 0x1

/**
 * p2p listen has started
 *
 * @see #ACTION_WIFI_P2P_LISTEN_STATE_CHANGED
 */

public static final int WIFI_P2P_LISTEN_STARTED = 2; // 0x2

/**
 * p2p listen has stopped
 *
 * @see #ACTION_WIFI_P2P_LISTEN_STATE_CHANGED
 */

public static final int WIFI_P2P_LISTEN_STOPPED = 1; // 0x1

/**
 * Broadcast intent action indicating that the available peer list has changed. This
 * can be sent as a result of peers being found, lost or updated.
 *
 * All of these permissions are required to receive this broadcast:
 * {@link android.Manifest.permission#ACCESS_WIFI_STATE} and either
 * {@link android.Manifest.permission#ACCESS_FINE_LOCATION} or
 * {@link android.Manifest.permission#NEARBY_WIFI_DEVICES}
 *
 * <p> An extra {@link #EXTRA_P2P_DEVICE_LIST} provides the full list of
 * current peers. The full list of peers can also be obtained any time with
 * {@link #requestPeers}.
 *
 * @see #EXTRA_P2P_DEVICE_LIST
 */

@android.annotation.SdkConstant(android.annotation.SdkConstant.SdkConstantType.BROADCAST_INTENT_ACTION) public static final java.lang.String WIFI_P2P_PEERS_CHANGED_ACTION = "android.net.wifi.p2p.PEERS_CHANGED";

/**
 * Run P2P scan on all channels.
 */

public static final int WIFI_P2P_SCAN_FULL = 0; // 0x0

/**
 * Run P2P scan only on a specific channel.
 */

public static final int WIFI_P2P_SCAN_SINGLE_FREQ = 2; // 0x2

/**
 * Run P2P scan only on social channels.
 */

public static final int WIFI_P2P_SCAN_SOCIAL = 1; // 0x1

/**
 * Broadcast intent action to indicate whether Wi-Fi p2p is enabled or disabled. An
 * extra {@link #EXTRA_WIFI_STATE} provides the state information as int.
 *
 * @see #EXTRA_WIFI_STATE
 */

@android.annotation.SdkConstant(android.annotation.SdkConstant.SdkConstantType.BROADCAST_INTENT_ACTION) public static final java.lang.String WIFI_P2P_STATE_CHANGED_ACTION = "android.net.wifi.p2p.STATE_CHANGED";

/**
 * Wi-Fi p2p is disabled.
 *
 * @see #WIFI_P2P_STATE_CHANGED_ACTION
 */

public static final int WIFI_P2P_STATE_DISABLED = 1; // 0x1

/**
 * Wi-Fi p2p is enabled.
 *
 * @see #WIFI_P2P_STATE_CHANGED_ACTION
 */

public static final int WIFI_P2P_STATE_ENABLED = 2; // 0x2

/**
 * Broadcast intent action indicating that this device details have changed.
 *
 * <p> An extra {@link #EXTRA_WIFI_P2P_DEVICE} provides this device details.
 * The valid device details can also be obtained with
 * {@link #requestDeviceInfo(android.net.wifi.p2p.WifiP2pManager.Channel,android.net.wifi.p2p.WifiP2pManager.DeviceInfoListener)} when p2p is enabled.
 * To get information notifications on P2P getting enabled refers
 * {@link #WIFI_P2P_STATE_ENABLED}.
 *
 * <p> The {@link #EXTRA_WIFI_P2P_DEVICE} extra contains an anonymized version of the device's
 * MAC address. Callers holding the {@code android.Manifest.permission#LOCAL_MAC_ADDRESS}
 * permission can use {@link #requestDeviceInfo} to obtain the actual MAC address of this
 * device.
 *
 * All of these permissions are required to receive this broadcast:
 * {@link android.Manifest.permission#ACCESS_WIFI_STATE} and either
 * {@link android.Manifest.permission#ACCESS_FINE_LOCATION} or
 * {@link android.Manifest.permission#NEARBY_WIFI_DEVICES}
 *
 * @see #EXTRA_WIFI_P2P_DEVICE
 */

@android.annotation.SdkConstant(android.annotation.SdkConstant.SdkConstantType.BROADCAST_INTENT_ACTION) public static final java.lang.String WIFI_P2P_THIS_DEVICE_CHANGED_ACTION = "android.net.wifi.p2p.THIS_DEVICE_CHANGED";
/** Interface for callback invocation on an application action */

@SuppressWarnings({"unchecked", "deprecation", "all"})
public static interface ActionListener {

/**
 * The operation failed
 * @param reason The reason for failure could be one of {@link #P2P_UNSUPPORTED},
 * {@link #ERROR} or {@link #BUSY}
 */

public void onFailure(int reason);

/** The operation succeeded */

public void onSuccess();
}

/**
 * A channel that connects the application to the Wifi p2p framework.
 * Most p2p operations require a Channel as an argument. An instance of Channel is obtained
 * by doing a call on {@link #initialize}
 */

@SuppressWarnings({"unchecked", "deprecation", "all"})
public static class Channel implements java.lang.AutoCloseable {

Channel() { throw new RuntimeException("Stub!"); }

/**
 * Close the current P2P connection and indicate to the P2P service that connections
 * created by the app can be removed.
 */

public void close() { throw new RuntimeException("Stub!"); }

/** @hide */

protected void finalize() throws java.lang.Throwable { throw new RuntimeException("Stub!"); }
}

/** Interface for callback invocation when framework channel is lost */

@SuppressWarnings({"unchecked", "deprecation", "all"})
public static interface ChannelListener {

/**
 * The channel to the framework has been disconnected.
 * Application could try re-initializing using {@link #initialize}
 */

public void onChannelDisconnected();
}

/** Interface for callback invocation when connection info is available */

@SuppressWarnings({"unchecked", "deprecation", "all"})
public static interface ConnectionInfoListener {

/**
 * The requested connection info is available
 * @param info Wi-Fi p2p connection info
 */

public void onConnectionInfoAvailable(android.net.wifi.p2p.WifiP2pInfo info);
}

/** Interface for callback invocation when {@link android.net.wifi.p2p.WifiP2pDevice}
 *  is available in response to {@link #requestDeviceInfo(android.net.wifi.p2p.WifiP2pManager.Channel,android.net.wifi.p2p.WifiP2pManager.DeviceInfoListener)}.
 */

@SuppressWarnings({"unchecked", "deprecation", "all"})
public static interface DeviceInfoListener {

/**
 * The requested {@link android.net.wifi.p2p.WifiP2pDevice} is available.
 * @param wifiP2pDevice Wi-Fi p2p {@link android.net.wifi.p2p.WifiP2pDevice}

 * This value may be {@code null}.
 */

public void onDeviceInfoAvailable(@android.annotation.Nullable android.net.wifi.p2p.WifiP2pDevice wifiP2pDevice);
}

/** Interface for callback invocation when p2p state is available
 *  in response to {@link #requestDiscoveryState}.
 */

@SuppressWarnings({"unchecked", "deprecation", "all"})
public static interface DiscoveryStateListener {

/**
 * The requested p2p discovery state is available.
 * @param state Wi-Fi p2p discovery state
 *
 * Value is {@link android.net.wifi.p2p.WifiP2pManager#WIFI_P2P_DISCOVERY_STOPPED}, or {@link android.net.wifi.p2p.WifiP2pManager#WIFI_P2P_DISCOVERY_STARTED}
 * @see #WIFI_P2P_DISCOVERY_STARTED
 *        @see #WIFI_P2P_DISCOVERY_STOPPED
 */

public void onDiscoveryStateAvailable(int state);
}

/**
 * Interface for callback invocation when Bonjour service discovery response
 * is received
 */

@SuppressWarnings({"unchecked", "deprecation", "all"})
public static interface DnsSdServiceResponseListener {

/**
 * The requested Bonjour service response is available.
 *
 * <p>This function is invoked when the device with the specified Bonjour
 * registration type returned the instance name.
 * @param instanceName instance name.<br>
 *  e.g) "MyPrinter".
 * @param registrationType <br>
 * e.g) "_ipp._tcp.local."
 * @param srcDevice source device.
 */

public void onDnsSdServiceAvailable(java.lang.String instanceName, java.lang.String registrationType, android.net.wifi.p2p.WifiP2pDevice srcDevice);
}

/**
 * Interface for callback invocation when Bonjour TXT record is available
 * for a service
 */

@SuppressWarnings({"unchecked", "deprecation", "all"})
public static interface DnsSdTxtRecordListener {

/**
 * The requested Bonjour service response is available.
 *
 * <p>This function is invoked when the device with the specified full
 * service domain service returned TXT record.
 *
 * @param fullDomainName full domain name. <br>
 * e.g) "MyPrinter._ipp._tcp.local.".
 * @param txtRecordMap TXT record data as a map of key/value pairs
 * @param srcDevice source device.
 */

public void onDnsSdTxtRecordAvailable(java.lang.String fullDomainName, java.util.Map<java.lang.String,java.lang.String> txtRecordMap, android.net.wifi.p2p.WifiP2pDevice srcDevice);
}

/**
 * Interface for callback invocation when an incoming request is received.
 *
 * This callback is registered by
 * {@link #addExternalApprover(android.net.wifi.p2p.WifiP2pManager.Channel,android.net.MacAddress,android.net.wifi.p2p.WifiP2pManager.ExternalApproverRequestListener)}.
 */

@SuppressWarnings({"unchecked", "deprecation", "all"})
public static interface ExternalApproverRequestListener {

/**
 * Called when an approver registration via
 * {@link #addExternalApprover(android.net.wifi.p2p.WifiP2pManager.Channel,android.net.MacAddress,android.net.wifi.p2p.WifiP2pManager.ExternalApproverRequestListener)}
 * is successful.
 *
 * @param deviceAddress is the peer MAC address used in the registration.

 * This value cannot be {@code null}.
 */

public void onAttached(@android.annotation.NonNull android.net.MacAddress deviceAddress);

/**
 * Called when there is an incoming connection request
 * which matches a peer (identified by its {@link android.net.MacAddress MacAddress}) registered by the external
 * approver through
 * {@link #addExternalApprover(android.net.wifi.p2p.WifiP2pManager.Channel,android.net.MacAddress,android.net.wifi.p2p.WifiP2pManager.ExternalApproverRequestListener)}.
 * The external approver is expected to follow up with a connection decision using the
 * {@link #setConnectionRequestResult(android.net.wifi.p2p.WifiP2pManager.Channel,android.net.MacAddress,int,android.net.wifi.p2p.WifiP2pManager.ActionListener)} with
 * {@link #CONNECTION_REQUEST_ACCEPT}, {@link #CONNECTION_REQUEST_REJECT}, or
 * {@link #CONNECTION_REQUEST_DEFER_TO_SERVICE}.
 *
 * @param requestType is one of {@link #REQUEST_TYPE_NEGOTIATION},
 *        {@link #REQUEST_TYPE_INVITATION}, and {@link #REQUEST_TYPE_JOIN}.
 * Value is {@link android.net.wifi.p2p.WifiP2pManager.ExternalApproverRequestListener#REQUEST_TYPE_NEGOTIATION}, {@link android.net.wifi.p2p.WifiP2pManager.ExternalApproverRequestListener#REQUEST_TYPE_INVITATION}, or {@link android.net.wifi.p2p.WifiP2pManager.ExternalApproverRequestListener#REQUEST_TYPE_JOIN}
 * @param config is the peer configuration.
 * This value cannot be {@code null}.
 * @param device is the peer information.

 * This value cannot be {@code null}.
 */

public void onConnectionRequested(int requestType, @android.annotation.NonNull android.net.wifi.p2p.WifiP2pConfig config, @android.annotation.NonNull android.net.wifi.p2p.WifiP2pDevice device);

/**
 * Called when an approver registration via
 * {@link #addExternalApprover(android.net.wifi.p2p.WifiP2pManager.Channel,android.net.MacAddress,android.net.wifi.p2p.WifiP2pManager.ExternalApproverRequestListener)}
 * has failed.
 *
 * @param deviceAddress is the peer MAC address used in the registration.
 * This value cannot be {@code null}.
 * @param reason is the failure reason.

 * Value is {@link android.net.wifi.p2p.WifiP2pManager.ExternalApproverRequestListener#APPROVER_DETACH_REASON_REMOVE}, {@link android.net.wifi.p2p.WifiP2pManager.ExternalApproverRequestListener#APPROVER_DETACH_REASON_FAILURE}, {@link android.net.wifi.p2p.WifiP2pManager.ExternalApproverRequestListener#APPROVER_DETACH_REASON_REPLACE}, or {@link android.net.wifi.p2p.WifiP2pManager.ExternalApproverRequestListener#APPROVER_DETACH_REASON_CLOSE}
 */

public void onDetached(@android.annotation.NonNull android.net.MacAddress deviceAddress, int reason);

/**
 * Called when a PIN is generated by the WiFi service.
 *
 * The external approver can display the PIN, exchange the PIN via Out-Of-Band way
 * or ask the wifi service to show the PIN as usual using the
 * {@link #setConnectionRequestResult(android.net.wifi.p2p.WifiP2pManager.Channel,android.net.MacAddress,int,android.net.wifi.p2p.WifiP2pManager.ActionListener)}
 * with {@link #CONNECTION_REQUEST_DEFER_SHOW_PIN_TO_SERVICE}.
 *
 * @param deviceAddress is the peer MAC address used in the registration.
 * This value cannot be {@code null}.
 * @param pin is the WPS PIN.

 * This value cannot be {@code null}.
 */

public void onPinGenerated(@android.annotation.NonNull android.net.MacAddress deviceAddress, @android.annotation.NonNull java.lang.String pin);

/**
 * Detached since the {@link android.net.wifi.p2p.WifiP2pManager WifiP2pManager} channel was closed, e.g.
 * by using {@link android.net.wifi.p2p.WifiP2pManager.Channel#close() Channel.close()} method.
 *
 * Used in {@link #onDetached(android.net.MacAddress,int)}.
 */

public static final int APPROVER_DETACH_REASON_CLOSE = 3; // 0x3

/**
 * Detached due to a framework failure.
 *
 * Used in {@link #onDetached(android.net.MacAddress,int)}.
 */

public static final int APPROVER_DETACH_REASON_FAILURE = 1; // 0x1

/**
 * Detached by a call to
 * {@link #removeExternalApprover(android.net.wifi.p2p.WifiP2pManager.Channel,android.net.MacAddress,android.net.wifi.p2p.WifiP2pManager.ActionListener)}.
 *
 * Used in {@link #onDetached(android.net.MacAddress,int)}.
 */

public static final int APPROVER_DETACH_REASON_REMOVE = 0; // 0x0

/**
 * Detached when a new approver replaces an old one.
 *
 * Used in {@link #onDetached(android.net.MacAddress,int)}.
 */

public static final int APPROVER_DETACH_REASON_REPLACE = 2; // 0x2

/**
 * This device received an invitation request from GO to join the group.
 *
 * Used in {@link #onConnectionRequested(int,android.net.wifi.p2p.WifiP2pConfig,android.net.wifi.p2p.WifiP2pDevice)}.
 */

public static final int REQUEST_TYPE_INVITATION = 1; // 0x1

/**
 * This GO device received a request from a peer to join the group.
 *
 * Used in {@link #onConnectionRequested(int,android.net.wifi.p2p.WifiP2pConfig,android.net.wifi.p2p.WifiP2pDevice)}.
 */

public static final int REQUEST_TYPE_JOIN = 2; // 0x2

/**
 * This device received a negotiation request from another peer.
 *
 * Used in {@link #onConnectionRequested(int,android.net.wifi.p2p.WifiP2pConfig,android.net.wifi.p2p.WifiP2pDevice)}.
 */

public static final int REQUEST_TYPE_NEGOTIATION = 0; // 0x0
}

/** Interface for callback invocation when group info is available */

@SuppressWarnings({"unchecked", "deprecation", "all"})
public static interface GroupInfoListener {

/**
 * The requested p2p group info is available
 * @param group Wi-Fi p2p group info
 */

public void onGroupInfoAvailable(android.net.wifi.p2p.WifiP2pGroup group);
}

/** Interface for callback invocation when {@link android.net.NetworkInfo} is available
 *  in response to {@link #requestNetworkInfo}.
 */

@SuppressWarnings({"unchecked", "deprecation", "all"})
public static interface NetworkInfoListener {

/**
 * The requested {@link android.net.NetworkInfo} is available
 * @param networkInfo Wi-Fi p2p {@link android.net.NetworkInfo}

 * This value cannot be {@code null}.
 */

public void onNetworkInfoAvailable(@android.annotation.NonNull android.net.NetworkInfo networkInfo);
}

/** Interface for callback invocation when p2p state is available
 *  in response to {@link #requestP2pState}.
 */

@SuppressWarnings({"unchecked", "deprecation", "all"})
public static interface P2pStateListener {

/**
 * The requested p2p state is available.
 * @param state Wi-Fi p2p state
 *
 * Value is {@link android.net.wifi.p2p.WifiP2pManager#WIFI_P2P_STATE_DISABLED}, or {@link android.net.wifi.p2p.WifiP2pManager#WIFI_P2P_STATE_ENABLED}
 * @see #WIFI_P2P_STATE_DISABLED
 *        @see #WIFI_P2P_STATE_ENABLED
 */

public void onP2pStateAvailable(int state);
}

/** Interface for callback invocation when peer list is available */

@SuppressWarnings({"unchecked", "deprecation", "all"})
public static interface PeerListListener {

/**
 * The requested peer list is available
 * @param peers List of available peers
 */

public void onPeersAvailable(android.net.wifi.p2p.WifiP2pDeviceList peers);
}

/**
 * Interface for callback invocation when service discovery response other than
 * Upnp or Bonjour is received
 */

@SuppressWarnings({"unchecked", "deprecation", "all"})
public static interface ServiceResponseListener {

/**
 * The requested service response is available.
 *
 * @param protocolType protocol type. currently only
 * {@link android.net.wifi.p2p.nsd.WifiP2pServiceInfo#SERVICE_TYPE_VENDOR_SPECIFIC WifiP2pServiceInfo.SERVICE_TYPE_VENDOR_SPECIFIC}.
 * @param responseData service discovery response data based on the requested
 *  service protocol type. The format depends on the service type.
 * @param srcDevice source device.
 */

public void onServiceAvailable(int protocolType, byte[] responseData, android.net.wifi.p2p.WifiP2pDevice srcDevice);

/**
 * The requested USD based service response is available.
 * @param srcDevice source device.
 * This value cannot be {@code null}.
 * @param usdResponseData {@link android.net.wifi.p2p.nsd.WifiP2pUsdBasedServiceResponse WifiP2pUsdBasedServiceResponse}.

 * This value cannot be {@code null}.
 */

public default void onUsdBasedServiceAvailable(@android.annotation.NonNull android.net.wifi.p2p.WifiP2pDevice srcDevice, @android.annotation.NonNull android.net.wifi.p2p.nsd.WifiP2pUsdBasedServiceResponse usdResponseData) { throw new RuntimeException("Stub!"); }
}

/**
 * Interface for callback invocation when upnp service discovery response
 * is received
 * */

@SuppressWarnings({"unchecked", "deprecation", "all"})
public static interface UpnpServiceResponseListener {

/**
 * The requested upnp service response is available.
 *
 * <p>This function is invoked when the specified device or service is found.
 *
 * @param uniqueServiceNames The list of unique service names.<br>
 * e.g) uuid:6859dede-8574-59ab-9332-123456789012::urn:schemas-upnp-org:device:
 * MediaServer:1
 * @param srcDevice source device.
 */

public void onUpnpServiceAvailable(java.util.List<java.lang.String> uniqueServiceNames, android.net.wifi.p2p.WifiP2pDevice srcDevice);
}

/**
 * Interface used to listen to Wi-Fi p2p various changes such as device state change,
 * discovery started/stopped, connection change, etc.
 */

@SuppressWarnings({"unchecked", "deprecation", "all"})
public static interface WifiP2pListener {

/**
 * Called when this device details have changed.
 * @see #WIFI_P2P_THIS_DEVICE_CHANGED_ACTION
 * @see #requestDeviceInfo(Channel, DeviceInfoListener)
 *
 * @param p2pDevice provides this device details.

 * This value may be {@code null}.
 */

public default void onDeviceConfigurationChanged(@android.annotation.Nullable android.net.wifi.p2p.WifiP2pDevice p2pDevice) { throw new RuntimeException("Stub!"); }

/**
 * Called when peer discovery has either started or stopped.
 * @see #WIFI_P2P_DISCOVERY_CHANGED_ACTION
 * @see #requestDiscoveryState(Channel, DiscoveryStateListener)
 *
 * @param state indicates whether discovery has started or stopped.
 *
 * Value is {@link android.net.wifi.p2p.WifiP2pManager#WIFI_P2P_DISCOVERY_STOPPED}, or {@link android.net.wifi.p2p.WifiP2pManager#WIFI_P2P_DISCOVERY_STARTED}
 * @see #WIFI_P2P_DISCOVERY_STARTED
 *          @see #WIFI_P2P_DISCOVERY_STOPPED
 */

public default void onDiscoveryStateChanged(int state) { throw new RuntimeException("Stub!"); }

/**
 * Called when the frequency of a formed group has been changed.
 *
 * @param p2pInfo  provides the p2p connection info.
 * This value cannot be {@code null}.
 * @param p2pGroup provides the details of the group.

 * This value cannot be {@code null}.
 */

public default void onFrequencyChanged(@android.annotation.NonNull android.net.wifi.p2p.WifiP2pInfo p2pInfo, @android.annotation.NonNull android.net.wifi.p2p.WifiP2pGroup p2pGroup) { throw new RuntimeException("Stub!"); }

/**
 * Called when either group owner or group client has created p2p group successfully.
 *
 * @param p2pInfo  provides the p2p connection info.
 * This value cannot be {@code null}.
 * @param p2pGroup provides the details of the group.

 * This value cannot be {@code null}.
 */

public default void onGroupCreated(@android.annotation.NonNull android.net.wifi.p2p.WifiP2pInfo p2pInfo, @android.annotation.NonNull android.net.wifi.p2p.WifiP2pGroup p2pGroup) { throw new RuntimeException("Stub!"); }

/**
 * Called when either group owner or group client is creating p2p group.
 */

public default void onGroupCreating() { throw new RuntimeException("Stub!"); }

/**
 * Called when group creation has failed.
 *
 * @param reason provides the group creation failure reason.

 * Value is {@link android.net.wifi.p2p.WifiP2pManager#GROUP_CREATION_FAILURE_REASON_CONNECTION_CANCELLED}, {@link android.net.wifi.p2p.WifiP2pManager#GROUP_CREATION_FAILURE_REASON_TIMED_OUT}, {@link android.net.wifi.p2p.WifiP2pManager#GROUP_CREATION_FAILURE_REASON_USER_REJECTED}, {@link android.net.wifi.p2p.WifiP2pManager#GROUP_CREATION_FAILURE_REASON_PROVISION_DISCOVERY_FAILED}, {@link android.net.wifi.p2p.WifiP2pManager#GROUP_CREATION_FAILURE_REASON_GROUP_REMOVED}, or {@link android.net.wifi.p2p.WifiP2pManager#GROUP_CREATION_FAILURE_REASON_INVITATION_FAILED}
 */

public default void onGroupCreationFailed(int reason) { throw new RuntimeException("Stub!"); }

/**
 * Called when group negotiation has been rejected by user.
 */

public default void onGroupNegotiationRejectedByUser() { throw new RuntimeException("Stub!"); }

/**
 * Called when p2p group has been removed.
 */

public default void onGroupRemoved() { throw new RuntimeException("Stub!"); }

/**
 * Called when peer listen has either started or stopped.
 * @see #ACTION_WIFI_P2P_LISTEN_STATE_CHANGED
 * @see #getListenState(Channel, Executor, Consumer)
 *
 * @param state indicates whether listen has started or stopped.
 *
 * Value is {@link android.net.wifi.p2p.WifiP2pManager#WIFI_P2P_LISTEN_STOPPED}, or {@link android.net.wifi.p2p.WifiP2pManager#WIFI_P2P_LISTEN_STARTED}
 * @see #WIFI_P2P_LISTEN_STARTED
 *          @see #WIFI_P2P_LISTEN_STOPPED
 */

public default void onListenStateChanged(int state) { throw new RuntimeException("Stub!"); }

/**
 * Called when Wi-Fi p2p has been enabled or disabled.
 * @see #WIFI_P2P_STATE_CHANGED_ACTION
 * @see #requestP2pState(Channel, P2pStateListener)
 *
 * @param state indicates whether Wi-Fi p2p is enabled or disabled.
 *
 * Value is {@link android.net.wifi.p2p.WifiP2pManager#WIFI_P2P_STATE_DISABLED}, or {@link android.net.wifi.p2p.WifiP2pManager#WIFI_P2P_STATE_ENABLED}
 * @see #WIFI_P2P_STATE_ENABLED
 *          @see #WIFI_P2P_STATE_DISABLED
 */

public default void onP2pStateChanged(int state) { throw new RuntimeException("Stub!"); }

/**
 * Called to indicate group owner that a group client has disconnected.
 *
 * @param p2pInfo  provides the p2p connection info.
 * This value cannot be {@code null}.
 * @param p2pGroup provides the details of the group.

 * This value cannot be {@code null}.
 */

public default void onPeerClientDisconnected(@android.annotation.NonNull android.net.wifi.p2p.WifiP2pInfo p2pInfo, @android.annotation.NonNull android.net.wifi.p2p.WifiP2pGroup p2pGroup) { throw new RuntimeException("Stub!"); }

/**
 * Called to indicate group owner that a group client has joined p2p group successfully.
 *
 * @param p2pInfo  provides the p2p connection info.
 * This value cannot be {@code null}.
 * @param p2pGroup provides the details of the group.

 * This value cannot be {@code null}.
 */

public default void onPeerClientJoined(@android.annotation.NonNull android.net.wifi.p2p.WifiP2pInfo p2pInfo, @android.annotation.NonNull android.net.wifi.p2p.WifiP2pGroup p2pGroup) { throw new RuntimeException("Stub!"); }

/**
 * Called when the available peer list has changed. This can be sent as a result of peers
 * being found, lost or updated.
 * @see #WIFI_P2P_PEERS_CHANGED_ACTION
 * @see #requestPeers(Channel, PeerListListener)
 *
 * @param p2pDeviceList provides the full list of current peers.

 * This value cannot be {@code null}.
 */

public default void onPeerListChanged(@android.annotation.NonNull android.net.wifi.p2p.WifiP2pDeviceList p2pDeviceList) { throw new RuntimeException("Stub!"); }
}

}

