/*
 * Copyright (C) 2014 The Android Open Source Project
 * Copyright (c) 2000, 2021, Oracle and/or its affiliates. All rights reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This code is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 only, as
 * published by the Free Software Foundation.  Oracle designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Oracle in the LICENSE file that accompanied this code.
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 *
 * You should have received a copy of the GNU General Public License version
 * 2 along with this work; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * Please contact Oracle, 500 Oracle Parkway, Redwood Shores, CA 94065 USA
 * or visit www.oracle.com if you need additional information or have any
 * questions.
 */


package java.nio;

import java.io.UncheckedIOException;

/**
 * A direct byte buffer whose content is a memory-mapped region of a file.
 *
 * <p> Mapped byte buffers are created via the {@link
 * java.nio.channels.FileChannel#map FileChannel.map} method.  This class
 * extends the {@link java.nio.ByteBuffer ByteBuffer} class with operations that are specific to
 * memory-mapped file regions.
 *
 * <p> A mapped byte buffer and the file mapping that it represents remain
 * valid until the buffer itself is garbage-collected.
 *
 * <p> The content of a mapped byte buffer can change at any time, for example
 * if the content of the corresponding region of the mapped file is changed by
 * this program or another.  Whether or not such changes occur, and when they
 * occur, is operating-system dependent and therefore unspecified.
 *
 * <a id="inaccess"></a><p> All or part of a mapped byte buffer may become
 * inaccessible at any time, for example if the mapped file is truncated.  An
 * attempt to access an inaccessible region of a mapped byte buffer will not
 * change the buffer's content and will cause an unspecified exception to be
 * thrown either at the time of the access or at some later time.  It is
 * therefore strongly recommended that appropriate precautions be taken to
 * avoid the manipulation of a mapped file by this program, or by a
 * concurrently running program, except to read or write the file's content.
 *
 * <p> Mapped byte buffers otherwise behave no differently than ordinary direct
 * byte buffers. </p>
 *
 *
 * @author Mark Reinhold
 * @author JSR-51 Expert Group
 * @since 1.4
 */

@SuppressWarnings({"unchecked", "deprecation", "all"})
public abstract class MappedByteBuffer extends java.nio.ByteBuffer {

MappedByteBuffer() { throw new RuntimeException("Stub!"); }

/**
 * Tells whether or not this buffer's content is resident in physical
 * memory.
 *
 * <p> A return value of {@code true} implies that it is highly likely
 * that all of the data in this buffer is resident in physical memory and
 * may therefore be accessed without incurring any virtual-memory page
 * faults or I/O operations.  A return value of {@code false} does not
 * necessarily imply that the buffer's content is not resident in physical
 * memory.
 *
 * <p> The returned value is a hint, rather than a guarantee, because the
 * underlying operating system may have paged out some of the buffer's data
 * by the time that an invocation of this method returns.  </p>
 *
 * @return  {@code true} if it is likely that this buffer's content
 *          is resident in physical memory
 */

public final boolean isLoaded() { throw new RuntimeException("Stub!"); }

/**
 * Loads this buffer's content into physical memory.
 *
 * <p> This method makes a best effort to ensure that, when it returns,
 * this buffer's content is resident in physical memory.  Invoking this
 * method may cause some number of page faults and I/O operations to
 * occur. </p>
 *
 * @return  This buffer
 */

public final java.nio.MappedByteBuffer load() { throw new RuntimeException("Stub!"); }

/**
 * Forces any changes made to this buffer's content to be written to the
 * storage device containing the mapped file.  The region starts at index
 * zero in this buffer and is {@code capacity()} bytes.
 *
 * <p> If the file mapped into this buffer resides on a local storage
 * device then when this method returns it is guaranteed that all changes
 * made to the buffer since it was created, or since this method was last
 * invoked, will have been written to that device.
 *
 * <p> If the file does not reside on a local device then no such guarantee
 * is made.
 *
 * <p> If this buffer was not mapped in read/write mode ({@link
 * java.nio.channels.FileChannel.MapMode#READ_WRITE}) then
 * invoking this method may have no effect. In particular, the
 * method has no effect for buffers mapped in read-only or private
 * mapping modes. This method may or may not have an effect for
 * implementation-specific mapping modes. </p>
 *
 * @throws java.io.UncheckedIOException
 *         If an I/O error occurs writing the buffer's content to the
 *         storage device containing the mapped file
 *
 * @return  This buffer
 */

public final java.nio.MappedByteBuffer force() { throw new RuntimeException("Stub!"); }

/**
 * Forces any changes made to a region of this buffer's content to
 * be written to the storage device containing the mapped
 * file. The region starts at the given {@code index} in this
 * buffer and is {@code length} bytes.
 *
 * <p> If the file mapped into this buffer resides on a local
 * storage device then when this method returns it is guaranteed
 * that all changes made to the selected region buffer since it
 * was created, or since this method was last invoked, will have
 * been written to that device. The force operation is free to
 * write bytes that lie outside the specified region, for example
 * to ensure that data blocks of some device-specific granularity
 * are transferred in their entirety.
 *
 * <p> If the file does not reside on a local device then no such
 * guarantee is made.
 *
 * <p> If this buffer was not mapped in read/write mode ({@link
 * java.nio.channels.FileChannel.MapMode#READ_WRITE}) then
 * invoking this method may have no effect. In particular, the
 * method has no effect for buffers mapped in read-only or private
 * mapping modes. This method may or may not have an effect for
 * implementation-specific mapping modes. </p>
 *
 * @param  index
 *         The index of the first byte in the buffer region that is
 *         to be written back to storage; must be non-negative
 *         and less than {@code capacity()}
 *
 * @param  length
 *         The length of the region in bytes; must be non-negative
 *         and no larger than {@code capacity() - index}
 *
 * @throws java.lang.IndexOutOfBoundsException
 *         if the preconditions on the index and length do not
 *         hold.
 *
 * @throws java.io.UncheckedIOException
 *         If an I/O error occurs writing the buffer's content to the
 *         storage device containing the mapped file
 *
 * @return  This buffer
 *
 * @since 13
 */

public final java.nio.MappedByteBuffer force(int index, int length) { throw new RuntimeException("Stub!"); }

/**
 * {@inheritDoc}
 */

public final java.nio.Buffer position(int newPosition) { throw new RuntimeException("Stub!"); }

/**
 * {@inheritDoc}
 */

public final java.nio.Buffer limit(int newLimit) { throw new RuntimeException("Stub!"); }

/**
 * {@inheritDoc}
 */

public final java.nio.Buffer mark() { throw new RuntimeException("Stub!"); }

/**
 * {@inheritDoc}
 */

public final java.nio.Buffer reset() { throw new RuntimeException("Stub!"); }

/**
 * {@inheritDoc}
 */

public final java.nio.Buffer clear() { throw new RuntimeException("Stub!"); }

/**
 * {@inheritDoc}
 */

public final java.nio.Buffer flip() { throw new RuntimeException("Stub!"); }

/**
 * {@inheritDoc}
 */

public final java.nio.Buffer rewind() { throw new RuntimeException("Stub!"); }

/**
 * {@inheritDoc}
 *
 * <p> Reading bytes into physical memory by invoking {@code load()} on the
 * returned buffer, or writing bytes to the storage device by invoking
 * {@code force()} on the returned buffer, will only act on the sub-range
 * of this buffer that the returned buffer represents, namely
 * {@code [position(),limit())}.
 */

public abstract java.nio.ByteBuffer slice();

/**
 * {@inheritDoc}
 *
 * <p> Reading bytes into physical memory by invoking {@code load()} on the
 * returned buffer, or writing bytes to the storage device by invoking
 * {@code force()} on the returned buffer, will only act on the sub-range
 * of this buffer that the returned buffer represents, namely
 * {@code [index,index+length)}, where {@code index} and {@code length} are
 * assumed to satisfy the preconditions.
 */

public abstract java.nio.MappedByteBuffer slice(int index, int length);

/**
 * {@inheritDoc}
 */

public abstract java.nio.ByteBuffer duplicate();

/**
 * {@inheritDoc}
 */

public abstract java.nio.ByteBuffer compact();
}

